<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

// Include definition file
require dirname(__FILE__) . '/define.php';

/**
* VP Framework Plugin Helper class
* 
* @since 1.0.0
*/
class plgSystemVPFrameworkHelper
{
	protected $input;
	protected $option;
	protected $view;
	protected $task;
	protected $method;
	protected $layout;
	protected $tmpl;
	protected $type;
	protected $params;
	protected $data;
	protected $start_time;
	
	public static $instance = null;
	
	protected static $_template = null;
	protected static $_templates = array();
	protected static $_needFramework = null;
	protected static $_autoloadRegistered = false;
	protected static $_defaultTemplate = null;
	protected static $_defaultTemplateXml = null;

	/**
	* Construction method of plgSystemVPFrameworkHelper class
	* 
	* @return void
	*/
	public function __construct()
	{
		jimport('cms.html.html');

		$this->start_time  = microtime(true);
		$app               = JFactory::getApplication();
		$doc               = JFactory::getDocument();
		$this->input       = $app->input;
		$this->option      = strtolower($this->input->getCmd('option', ''));
		$this->view        = strtolower($this->input->getCmd('view', ''));
		$this->task        = $this->input->post->getCmd('task', '') ?
		                     strtolower($this->input->post->getCmd('task', '')) :
		                     strtolower($this->input->getCmd('task', 'display'));
		$this->method      = $this->input->post->getCmd('vpfmethod', '') ?
		                     strtolower($this->input->post->getCmd('vpfmethod', '')) :
		                     strtolower($this->input->getCmd('vpfmethod', ''));
		$this->layout      = strtolower($this->input->getCmd('layout', ''));
		$this->tmpl        = strtolower($this->input->getCmd('tmpl', ''));
		$this->type        = strtolower($doc->getType());
		
		if ($app->isClient('site'))
		{
			$template     = self::getTemplate();
			$this->params = $template->params;
		}
		
		if (!empty($this->task) && strpos($this->task, '.') !== false)
		{
			$parts      = explode('.', $this->task);
			$this->view = $parts[0];
			$this->task = $parts[1];
		}
		
		// Load plugin language
		$lang = JFactory::getLanguage();
		$lang->load('plg_system_vpframework', JPATH_ADMINISTRATOR, 'en-GB', true);
		$lang->load('plg_system_vpframework', JPATH_ADMINISTRATOR, $lang->getDefault(), true);
		$lang->load('plg_system_vpframework', JPATH_ADMINISTRATOR, null, true);
	}
	
	public static function getInstance($cache = true)
	{
		if (!$cache)
		{
			return new plgSystemVPFrameworkHelper();
		}
		
		if (self::$instance === null)
		{
			self::$instance = new plgSystemVPFrameworkHelper();
		}
		
		return self::$instance;
	}
	
	/**
	* Method to register autoloader function for all VP Framework helper class
	* 
	* @return void
	*/
	public static function autoloadRegister()
	{
		if (self::$_autoloadRegistered === false)
		{
			spl_autoload_register(array('plgSystemVPFrameworkHelper', 'helperLoader'));
			self::$_autoloadRegistered = true;
		}
		
		// Need a small hack to override core chromestyle field
		if (!class_exists('JFormFieldChromeStyle')) require VPF_PLUGINPATH . '/fields/chromestyle.php';
	}
	
	/**
	* Method to autoload all VP Framework helper class
	* 
	* @param string $class Name of the helper class
	* 
	* @return boolean True is sucess or fals if failure
	*/
	private static function helperLoader($class) 
	{
		// If class does not have VPFramework prefix then it is not our helper.
		if (strpos(strtolower($class), 'vpframework') !== 0)
		{
			return;
		}
		
		// If already loaded return true
		if (class_exists($class, false))
		{
			return true;
		}
		
		// Path of helpers directory
		$path      = VPF_PLUGINPATH . DIRECTORY_SEPARATOR . 'helpers';
		// Retrieve the filename from the class
		$filename  = strtolower(substr($class, 11)) . '.php';
		// Build full file path
		$file      = $path . DIRECTORY_SEPARATOR . $filename;
		
		if (is_file($file) && !class_exists($class)) 
		{
			// echo 'Trying to load ', $class, ' via ', __METHOD__, "()\n";
			// Lets try to load the helper file if class no exists			
			require_once($file);
			// If the class still does exists return false
			if (!class_exists($class))
			{
				return false;
			} 
		}
		// Class loaded. Return true.
		return true;
	}

	/**
	* Method to check if we need to use VP Framework
	* 
	* @param  boolean  $htmlOnly If true only valid for document type HTML.
	* 
	* @return boolean  False if not needed
	*/
	public function needVPF($htmlOnly = false)
	{
		$app = JFactory::getApplication();
		
		if ($htmlOnly && $this->type != 'html')
		{
			return false;
		}
		
		if ($app->isClient('administrator'))
		{
			$vpf_templates = $this->getTemplates();
			$needed = !empty($vpf_templates) ? true : false;
		}
		else
		{
			$template = self::getTemplate();
			$needed = $template->params->get('vpframework', 0) ? true : false;
		}
		
		return $needed;
	}
	
	public function initialize()
	{
		$app = JFactory::getApplication();
		
		if ($app->isClient('site'))
		{
			VPFrameworkFilter::initialiseFilter();
			
			$dlkHelper = VPFrameworkDlk::getInstance();
			
			if (!empty($dlkHelper->package) && empty($dlkHelper->value) && !empty($dlkHelper->template->template))
			{
				JFactory::getLanguage()->load('tpl_' . $dlkHelper->template->template);
				
				$templateName = JText::_(strtoupper($dlkHelper->template->template));
				$message = '5468616e6b20796f7520666f7220696e7374616c6c696e67203c623e25733c2f623e2074656d706c6174652e20506c656173652076616c696461746520796f757220636f7079206f66207468652074656d706c617465207061636b6167652e203c62722f3e476f20746f203c623e25733c2f623e2074656d706c6174652073657474696e677320696e204a6f6f6d6c612041646d696e6973747261746f7220616e642061646420796f7572203c623e446f776e6c6f6164204b65793c2f623e20756e646572203c623e4d656d6265727368697020496e666f726d6174696f6e3c2f623e2073656374696f6e2e';
				
				$app->enqueueMessage(sprintf(hex2bin($message), $templateName, $templateName), 'info');
			}
		}
	}
	
	/**
	* Method to get the VPFrameworkTemplate helper instance
	* 
	* @return object VPFrameworkTemplate class object
	*/
	public static function getTemplate()
	{
		if (static::$_template === null)
		{
			static::$_template = VPFrameworkTemplate::getInstance();
		}
		
		return static::$_template;
	}
	
	/**
	* Method to get all installed VPFramework based templates
	* 
	* @param string  $field_name   Optional name of the column if need only a specific field from template object.
	* @param integer $client_id    Cient ID
	* @param booleab $enabled_only If only enabled then set this as true.
	* 
	* @return mixed (boolean/string/integer/object) False in case of failure.
	*/
	public function getTemplates($field_name = null, $client_id = 0, $enabled_only = false)
	{
		if (!empty($field_name))
		{
			$field_name = (string) $field_name;
			if (!in_array($field_name, array('id', 'template', 'home', 'params')))
			{
				throw new Exception('plgSystemVPFrameworkHelper::getTemplates Error: Invalid field name', 403);
				return false;
			}
		}
		
		$client_id    = in_array($client_id, array(0, 1)) ? $client_id : 0;
		$enabled_only = $enabled_only ? (bool) $enabled_only : false;
		$key          = 'client_id:' . $client_id . '.enabled_only:' . $enabled_only;
		
		if (!isset(static::$_templates[$key]))
		{
			$db = JFactory::getDBO();
			$query = $db->getQuery(true)
			            ->select('a.id, a.template, a.home, a.params')
			            ->from('#__template_styles AS a')
			            ->where('a.client_id = ' . (int) $client_id);

			if ($enabled_only)
			{
				$query->select('b.manifest_cache')
				      ->join('LEFT', '#__extensions AS b ON b.element = a.template')
				      ->where('b.type = ' . $db->quote('template'))
				      ->where('b.enabled = 1');
			}
			
			try
			{
				$db->setQuery($query);
				static::$_templates[$key] = $db->loadObjectList();
			}
			catch (Exception $e) 
			{
				throw new Exception('plgSystemVPFrameworkHelper::getTemplates Error: ' . $e->getMessage(), 403);
				return false;
			}
		}
		
		if (empty(static::$_templates[$key]))
		{
			return false;
		}
		
		$return = array();
		
		foreach (static::$_templates[$key] as $template)
		{
			if (!empty($template->params))
			{
				$params = new JRegistry;
				$params->loadString($template->params);
				$template->params = $params;
				
				if ($params->get('vpframework', 0))
				{ 
					if (!empty($field_name) && isset($template->$field_name))
					{
						$return[] = $template->$field_name;
					}
					else
					{
						$return[] = $template;
					}
				}
			}
		}
		
		return $return;
	}
	
	/**
	* Method to get default template object
	* 
	* @param integer $client_id
	* 
	* @return mixed (object/false) False in case failure
	*/
	public function getDefaultTemplate($client_id = 0)
	{
		if (self::$_defaultTemplate === null)
		{
			$db = JFactory::getDBO();
			$query = $db->getQuery(true)
			            ->select('a.id, a.template, a.home, a.params, b.manifest_cache')
			            ->from('#__template_styles AS a')
			            ->where('a.client_id = ' . (int) $client_id)
			            ->where('a.home = 1')
			            ->join('LEFT', '#__extensions AS b ON b.element = a.template')
			            ->where('b.type = ' . $db->quote('template'))
			            ->where('b.enabled = 1');
			try
			{
				$db->setQuery($query);
				$template = $db->loadObject();
			}
			catch (Exception $e) 
			{
				throw new Exception('plgSystemVPFrameworkHelper::getDefaultTemplate Error: ' . $e->getMessage(), 403);
				return false;
			}
			
			self::$_defaultTemplate = !empty($template) ? $template : false;
		}
		
		return self::$_defaultTemplate;
	}
	
	/**
	* Method to get the default template's xml
	* 
	* @return mixed (object/false) False in case of failure
	*/
	public function getDefaultTemplateXml()
	{
		if (self::$_defaultTemplateXml === null)
		{
			if (!$template = $this->getDefaultTemplate())
			{
				return false;
			}
			
			$xmlPath = JPath::clean(JPATH_SITE . '/templates/' . $template->template . '/templateDetails.xml');
			
			self::$_defaultTemplateXml = simplexml_load_file($xmlPath);
		}
		
		return self::$_defaultTemplateXml;
	}

	/**
	* Method to get the list of forms to skip
	* 
	* @param string  $type        params or form
	* @param boolean $fullPath    True of full path
	* 
	* @return array
	*/
	public function getFormsToSkip($type = 'params', $fullPath = false)
	{
		jimport('joomla.filesystem.file');
		
		$xml = $this->getDefaultTemplateXml();
		$result = array();
		
		if ($xml instanceof JXMLElement && !empty($xml->ignores))
		{
			if (!empty($xml->ignores->$type))
			{
				$ignores = (array) $xml->ignores->$type;
				
				if (!$fullPath)
				{
					return $ignores;
				}
				
				$path = ($type == 'params') ? VPF_PLUGINPATH . '/forms/params' : VPF_PLUGINPATH . '/forms';
				
				foreach ($ignores as $ignore)
				{
					$file = JPath::clean($path . '/' . $ignore . '.xml');
					if (JFile::exists($file))
					{
						$result[] = $file;
					}
				}
			}
		}
		
		return $result;
	}

	/**
	* Method to load template admin edit layout and to call edit method
	* 
	* @return mixed (void/json object)
	*/
	public function loadTemplateAdmin()
	{
		if ($this->option == 'com_templates' && $this->view == 'style' && $this->layout == 'edit')
		{
			$vpf_template_ids = $this->getTemplates('id');
			$adminHelper      = VPFrameworkAdmin::getInstance();
			$template_id      = $this->input->getInt('id', 0);

			if ($this->task == 'display' && $this->type == 'html' && !empty($vpf_template_ids) && in_array($template_id, $vpf_template_ids))
			{
				if (version_compare(JVERSION, '4.0.0', 'ge'))
				{
					$app        = JFactory::getApplication();
					$basePath   = JPATH_BASE . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_templates';
					$layoutPath = VPF_PLUGINPATH . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'tmpl';
					$component  = $app->bootComponent($this->option);
					$controller = $component->getMVCFactory($app)->createController($this->view, 'Administrator', array('base_path' => $basePath, 'option' => 'com_templates'), $app, $app->input);
					//$view       = $component->getMVCFactory($app)->createView($this->view, 'Administrator', $this->type, array('base_path' => $basePath, 'option' => 'com_templates', 'template_path' => $layoutPath));
					$view       = $controller->getView($this->view, $this->type, 'Administrator', array('base_path' => $basePath, 'option' => 'com_templates', 'template_path' => $layoutPath, 'layout' => 'default'));
				}
				else
				{
					$componenPath = JPATH_BASE . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_templates';
					$layoutPath   = VPF_PLUGINPATH . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'tmpl';
					$controller   = JControllerLegacy::getInstance('Templates', array('base_path' => $componenPath, 'name' => 'Templates', 'format' => $this->type));
					
					if ($controller)
					{
						$view = $controller->getView($this->view, $this->type, '', array('base_path' => $componenPath, 'template_path' => $layoutPath, 'layout' => $this->layout));
					}
				}
			}
			elseif (!empty($vpf_template_ids) && !empty($template_id) && in_array($template_id, $vpf_template_ids) && method_exists($adminHelper, $this->task))
			{
				return call_user_func(array($adminHelper, $this->task));
			}
		}
	}
	
	/**
	* Method to save additional template data
	* 
	* @return void
	*/
	public function onExtensionBeforeSave($context, $table, $isNew)
	{
		if ($context == 'com_templates.style')
		{
			if (!empty($table) && is_object($table) && !empty($table->client_id))
			{
				return;
			}
			
			$isOur = true;
			
			if (!empty($table) && is_object($table) && !empty($table->params))
			{
				try
				{
					$params = new JRegistry($table->params);
				}
				catch (RuntimeException $e)
				{
					// Do nothing
				}
				
				if (!empty($params) && !$params->get('vpframework', 0))
				{
					$isOur = false;
				}
			}
			
			if ($isOur)
			{
				$vpf_template_ids = $this->getTemplates('id');
				$template_id      = $this->input->getInt('id', 0);
				
				if (empty($vpf_template_ids))
				{
					$isOur = false;
				}
				elseif (!empty($template_id) && !in_array($template_id, $vpf_template_ids))
				{
					$isOur = false;
				}
			}

			if ($isOur)
			{
				$post     = $this->input->post->get('jform', array(), 'array');
				$data     = !empty($post['less']) ? $post['less'] : array();
				$theme    = isset($post['params']) && isset($post['params']['theme']) ? $post['params']['theme'] : null;
				$newTheme = isset($post['params']) && isset($post['params']['new_theme']) ? $post['params']['new_theme'] : false;
				
				if (!empty($theme) && !empty($data))
				{
					return VPFrameworkAdmin::getInstance()->saveTheme($theme, $data, $newTheme, $table);
				}
			}
		}
	}

	/**
	* Method to register VPF Methods
	* 
	* @return mixed
	*/
	public function registerVPFMethods()
	{
		if (!empty($this->method))
		{
			$methodHelper = VPFrameworkMethod::getInstance();
			if (method_exists($methodHelper, $this->method))
			{
				return call_user_func(array($methodHelper, $this->method));
			}
		}
	}

	/**
	* Method to set VirtueMart Category Product Listing View Mode
	* 
	* @return void
	*/
	public function setVMProductListMode()
	{
		if (($this->option == 'com_virtuemart' && $this->view == 'category') || ($this->option == 'com_customfilters' && $this->view == 'products'))
		{
			if (!class_exists('VmConfig'))
			{
				require(JPATH_ADMINISTRATOR . '/components/com_virtuemart/helpers/config.php');
			}
			
			if (!class_exists('ShopFunctionsF'))
			{
				require(JPATH_SITE . '/components/com_virtuemart/helpers/shopfunctionsf.php');
			}
			
			VmConfig::loadConfig();
			
			$app               = JFactory::getApplication();
			$menus             = $app->getMenu();
			$menu              = $menus->getActive();
			$sublayout         = isset($menu->query['productsublayout']) ? $menu->query['productsublayout'] : VmConfig::get('productsublayout', 'products');
			$category_id       = vRequest::getInt('virtuemart_category_id', -1);
			$manufacturer_id   = vRequest::getInt('virtuemart_manufacturer_id', -1 );
			
			if ($category_id === -1 && !empty($menu->query['virtuemart_category_id']))
			{
				$category_id = $menu->query['virtuemart_category_id'];
			}
			elseif ($category_id === -1 && $manufacturer_id === -1)
			{
				$category_id = ShopFunctionsF::getLastVisitedCategoryId();
			}
			
			$category_id = ($category_id === -1) ? 0 : $category_id;
			$paramPrefix = (isset($menu->query['virtuemart_category_id']) && $menu->query['virtuemart_category_id'] != $category_id) ? 'stf_' : '';
			$menuParams  = null;
			
			if (is_object($menu))
			{
				$menuParams = method_exists($menu, 'getParams') ? $menu->getParams() : $menu->params;
			}
			
			$sublayout        = !empty($menuParams) && $menuParams->get($paramPrefix . 'productsublayout', 0) ? $menuParams->get($paramPrefix . 'productsublayout', 0) : $sublayout;
			$modeMap          = array('products' => 'grid', 'products_horizon' => 'list', 'products_slim' => 'list');
			$defaultMode      = array_key_exists($sublayout, $modeMap) ? $modeMap[$sublayout] : null;
			$category_ids     = $app->input->get('virtuemart_category_id', array(0), 'ARRAY');
			$category_ids     = array_map('intval', $category_ids);
			$category_id_hash = ($this->option == 'com_customfilters' && $this->view == 'products') ? 'customfilters' : md5(serialize($category_ids));
			$viewMode         = $app->getUserStateFromRequest('vpframework.vm.viewmode.category.' . $category_id_hash, 'viewmode', $defaultMode, 'CMD');

			
			if ($defaultMode)
			{
				if ($sublayout == 'products_horizon')
				{
					unset($modeMap['products_slim']);
				}
				elseif ($sublayout == 'products_slim')
				{
					unset($modeMap['products_horizon']);
				}
			}
			
			$activeSublayout = in_array($viewMode, $modeMap) ? array_search($viewMode, $modeMap) : $sublayout;
			$viewModeInput   = strtolower($this->input->get('viewmode', '', 'WORD'));

			if (array_key_exists($activeSublayout, $modeMap) && $sublayout != $activeSublayout)
			{
				$menu->query['productsublayout'] = $activeSublayout;
				
				if ($menuParams)
				{
					$menuParams->set('productsublayout', $activeSublayout);
					$menuParams->set('stf_productsublayout', $activeSublayout);
				}
				
				if (empty($viewModeInput))
				{
					$app->setUserState('vpframework.vm.productsublayout.category.' . $category_id_hash, $activeSublayout);
				}
			}
			else
			{
				$app->setUserState('vpframework.vm.productsublayout.category.' . $category_id_hash, $sublayout);
			}
			
			if ($category_id > 0 && $this->params->get('vm_normalize_product_list', 1) && $menuParams)
			{
				$menuParams->set('featured', 0);
				$menuParams->set('stf_featured', 0);
				$menuParams->set('discontinued', 0);
				$menuParams->set('stf_discontinued', 0);
				$menuParams->set('latest', 0);
				$menuParams->set('stf_latest', 0);
				$menuParams->set('topten', 0);
				$menuParams->set('stf_topten', 0);
				$menuParams->set('recent', 0);
				$menuParams->set('stf_recent', 0);
				
				if (method_exists($menu, 'setParams'))
				{
					$menu->setParams($menuParams);
				}
			}
		}
	}
	
	/**
	* Method to prepare special Framework views
	* 
	* @return void
	*/
	public function prepareSpecialViews()
	{
		if ($this->type != 'html')
		{
			return;
		}
		
		// Override Content Controller Display function to ensure
		// Compare Page and Wishlist page are not cached
		VPFrameworkOverrides::process();
	}
	
	public function manageModuleCache()
	{
		$db = JFactory::getDbo();
		$needChange = array('mod_virtuemart_category', 'mod_virtuemart_search');
		$query = $db->getQuery(true)
		            ->select('title, module AS name')
		            ->from('#__modules')
		            ->where('published = 1')
		            ->where('module IN (' . implode(',', $db->quote($needChange)) . ')');
		$db->setQuery($query);
		
		try
		{
			$modules = $db->loadObjectList();
		}
		catch (RuntimeException $e)
		{
			JLog::add('plgSystemVPFrameworkHelper::manageModuleCache - ' . $e->getMessage(), JLog::WARNING, 'jerror');
			return;
		}
		
		if (empty($modules))
		{
			return;
		}
		
		jimport('cms.module.helper');
		
		foreach ($modules as $module)
		{
			$mod = JModuleHelper::getModule($module->name, $module->title);
			if (!empty($mod->id))
			{
				$params = new JRegistry;
				$params->loadString($mod->params);
				$params->set('owncache', 0);
				$params->set('cache', 0);
				$mod->params = $params->toString();
			}
		}

	}
	
	public function manageBreadcrumbs()
	{
		$template = $this->getTemplate();
		
		if ($template)
		{
			$template->initialisePageHeadingInBreadcrumbs();
		}
	}
	
	public function loadInlineAssets()
	{
		if ($this->type != 'html')
		{
			return;
		}
		
		$root = JUri::root(true);
		$path = JUri::root();
		$Itemid = $this->input->getInt('Itemid', 0);
		
		// Load Variables for Javascript
		VPFrameworkScript::addOption('URI', $root . '/index.php');
		VPFrameworkScript::addOption('AJAXURI', $root . '/index.php?option=com_ajax&plugin=vpframework&format=json');
		VPFrameworkScript::addOption('PATH', $path);
		VPFrameworkScript::addOption('TOKEN', JSession::getFormToken());
		VPFrameworkScript::addOption('ITEMID', $Itemid);

		// Load Language for Javascript
		JText::script('JALL'); 
		JText::script('JCANCEL'); 
		JText::script('JSAVE'); 
		JText::script('JSHOW'); 
		JText::script('JSUBMIT'); 
		JText::script('JYES'); 
		JText::script('JNO'); 
		JText::script('ERROR');
		JText::script('MESSAGE');
		JText::script('NOTICE');
		JText::script('WARNING');
		JText::script('JLIB_HTML_BEHAVIOR_CLOSE');
		JText::script('PLG_SYSTEM_VPFRAMEWORK_COMPARE');
		JText::script('PLG_SYSTEM_VPFRAMEWORK_WISHLIST_SHOW');
		
		if (VPFrameworkUtility::hasExtension('com_virtuemart', 'component'))
		{
			VPFrameworkVM::loadLanguages();
			JText::script('COM_VIRTUEMART_PRODUCT_ADDED_SUCCESSFULLY');
		}
	}
	
	/**
	* Method to prepare site HTML head
	* 
	* @return void
	*/
	public function prepareHead()
	{
		$doc            = JFactory::getDocument();
		$template       = self::getTemplate();
		$root           = JUri::root(true) . '/';
		$rootAbs        = JUri::root();
		$dsFromUri      = array(JPATH_BASE . DIRECTORY_SEPARATOR, DIRECTORY_SEPARATOR);
		$dsToUri        = array($root, '/');
		$scripts        = $doc->_scripts;
		$doc->_scripts  = array(); // Clear the existing scripts from queue
		$mootools       = array(
		                   'media/system/js/mootools-core.js',
		                   'media/system/js/mootools-core-uncompressed.js',
		                   'media/system/js/mootools-more.js',
		                   'media/system/js/mootools-more-uncompressed.js',
		                   'media/system/js/modal.js',
		                   'media/system/js/modal-uncompressed.js'
		                  );
		$jQueryMigrate  = array(
		                   $root . 'media/jui/js/jquery-migrate.min.js',
		                   $root . 'media/jui/js/jquery-migrate.js'
		                  );
		$unsetScripts   = array(
		                   $root . 'media/jui/js/bootstrap.min.js',
		                   $root . 'media/jui/js/bootstrap.js'
		                  );
		$temp           = array();
		$jQueryLoaded   = false;
		$autoCompleteJS = false;

		// Load inline options
		VPFrameworkScript::loadOptions();

		foreach ($scripts as $scriptSrc => $scriptAttr)
		{
			if (in_array($scriptSrc, $jQueryMigrate))
			{
				$jQueryLoaded = true;
			}
			elseif (strpos($scriptSrc, 'jquery-ui.min.js?vmver=') !== false)
			{
				$parts = explode('?', $scriptSrc);
				$scriptSrc = $parts[0];
			}
			elseif (strpos($scriptSrc, 'jquery.noconflict.js?vmver=') !== false)
			{
				continue;
			}
			elseif (in_array($scriptSrc, $unsetScripts) && !$this->_isFrontendEdit())
			{
				continue;
			}
			elseif (strpos($scriptSrc, 'jquery.autocomplete.min.js') !== false || strpos($scriptSrc, 'jquery.autocomplete.js') !== false)
			{
				$autoCompleteJS = $scriptSrc;
				continue;
			}
			
			$oPath = null;
			$src = $scriptSrc;
			
			if (!$this->_isFrontendEdit())
			{
				if (strpos($scriptSrc, $root) !== false)
				{
					$oPath = ($root != '/') ? JPATH_BASE . DIRECTORY_SEPARATOR . str_replace($root, '', $scriptSrc) : JPATH_BASE . DIRECTORY_SEPARATOR . $scriptSrc;
				}
				elseif (strpos($scriptSrc, $rootAbs) !== false)
				{
					$oPath = JPATH_BASE . DIRECTORY_SEPARATOR . str_replace($rootAbs, '', $scriptSrc);
				}
				
				if (!empty($oPath) && is_file(JPath::clean($oPath)))
				{
					$fPath = str_replace(JPATH_BASE . DIRECTORY_SEPARATOR, $template->media_path . '/js/', $oPath);
					
					if (is_file(JPath::clean($fPath)))
					{
						$src = str_replace($dsFromUri, $dsToUri, $fPath);
					}
				}
			}
			
			if (strpos($src, '//') !== 0 && strpos($src, 'http') !== 0)
			{
				$src = str_replace('//', '/', $src);
			}
			
			// Check for core Mootools scripts
			if ($this->strposa($src, $mootools) !== false && !$jQueryLoaded)
			{
				$temp[$src] = $scriptAttr;
				continue;
			}
			
			// Add back the scripts
			$this->addBackScript($doc, $src, $scriptAttr);
			
			if ($jQueryLoaded)
			{
				if (count(JText::script()) && !$this->_isFrontendEdit())
				{
					$doc->addScript($root . 'media/vpframework/js/media/system/js/core.js');
				}
				
				if (!empty($temp))
				{
					foreach ($temp as $script => $attribs)
					{
						$this->addBackScript($doc, $script, $attribs);
					}
					
					$temp = array();
				}
			}
		}
		
		if ($autoCompleteJS)
		{
			$doc->addScript($autoCompleteJS);
		}
		
		// Load Custom JavaScript file if any
		$template->loadSpecialJS();
		
		if (!$this->_isFrontendEdit())
		{
			/* Add Template Theme */
			$template->loadSpecialCSS();
			
			// Need to remove core VirtueMart CSS.
			$vmStyleSheets = array(
				'vm-ltr-common.css', 'vm-ltr-reviews.css',
				'vm-ltr-site.css', 'vmpanels.css',
				'vmsite-ltr.css', 'vmsite-rtl.css'
			);
			$vmStyleSheets = array_map('preg_quote', $vmStyleSheets);
			$pattern = implode('|', $vmStyleSheets);
			
			foreach ($doc->_styleSheets as $styleSheet => &$values)
			{
				if (preg_match("%\b/($pattern)\b%i", $styleSheet) > 0)
				{
					unset($doc->_styleSheets[$styleSheet]);
				}
			}
			
			// Remove Joomla Generator meta tag
			if ($this->params->get('optimizer_remove_generator', 1))
			{
				$doc->setGenerator(null);
			}
			
			if (VPFrameworkUtility::hasExtension('com_virtuemart', 'component'))
			{
				jimport( 'joomla.application.module.helper' );
				
				$modParams = new JRegistry;
				$module    = JModuleHelper::getModule('mod_vp_custom_filter');
				
				if (empty($module->id))
				{
					return;
				}

				$modParams    = new JRegistry($module->params);
				$filterHelper = VPFrameworkFilter::getInstance($this->params, $modParams);
				
				$filterHelper->prepareHead();
			}
		}
		
		// Add custom favicon
		$custom_favicon = $this->params->get('custom_favicon', '');
		
		if (!empty($custom_favicon))
		{
			if (strpos($custom_favicon, '#') !== false)
			{
				$parts          = explode('#', $custom_favicon);
				$custom_favicon = $parts[0];
			}
			
			foreach ($doc->_links as $url => $script)
			{
				if (strpos($url, '/favicon.ico') !== false)
				{
					unset($doc->_links[$url]);
				}
			}
			
			$doc->addFavicon($custom_favicon, 'image/x-icon',  'shortcut icon');
			
			if (version_compare(JVERSION, '4.0.0', 'ge'))
			{
				$doc->addHeadLink($custom_favicon, 'shortcut icon', 'rel', array('type' => 'image/vnd.microsoft.icon'));
			}
		}
	}
	
	protected function addBackScript(&$doc, $url, $info = array())
	{
		$mime  = isset($info['mime']) ? $info['mime'] : 'text/javascript';
		$defer = isset($info['defer']) ? $info['defer'] : false;
		$async = isset($info['async']) ? $info['async'] : false;
		
		if (version_compare(JVERSION, '3.7.0', 'ge'))
		{
			$attribs = array();
			$options = array();
			
			$attribs['type'] = $mime;
			$attribs['defer'] = $defer;
			$attribs['async'] = $async;
			
			if (isset($info['options']))
			{
				$options = $info['options'];
			}
			
			$doc->addScript($url, $options, $attribs);
		}
		else
		{
			$doc->addScript($url, $mime, $defer, $async);
		}
		
		return $doc;
	}
	
	protected function _isFrontendEdit()
	{
		$app = JFactory::getApplication();
		
		if ($app->isClient('administrator'))
		{
			return false;
		}
		
		return self::getTemplate()->_isFrontendEdit();
	}
	
	/**
	* strpos() where needles is array
	* 
	* @param string  $haystack  Haystack to check
	* @param array   $needles   Array of needles
	* @param integer $offset    Offset in the haystack
	* @param boolean $getNeedle Default should be 'false'. If the a call finds a match then you can retreve the matched 
	*                                                      needle by calling it again setting this value 'true'.
	* 
	* @return mixed (boolean/integer/string) False if not found or integer value of found position or found needle (string)
	*/
	protected function strposa($haystack, $needles, $offset=0, $getNeedle = false)
	{
		static $foundNeedles = array();
		$key = (string) $haystack . serialize($needles) . $offset;
		$foundNeedles[$key] = null;
		
		if ($getNeedle)
		{
			$foundNeedle = isset($foundNeedles[$key]) ? $foundNeedles[$key] : '';
			return $foundNeedle;
		}
		
		if (isset($foundNeedles[$key]))
		{
			return $foundNeedles[$key];
		}
		
		foreach ($needles as $needle)
		{
			$pos = strpos($haystack, $needle, $offset);
			if ($pos !== false)
			{
				$foundNeedles[$key] = $needle;
				return $pos;
			}
		}
		
		return false;
	}
	
	/**
	* Method to prepare body after render
	* 
	* @return void
	*/
	public function prepareBody()
	{
		$app = JFactory::getApplication();
		
		// Get body buffer
		$buffer = $app->getBody();
		
		// Remove VirtueMart Chosen JavaScript
		$buffer = preg_replace("/<script id=\"updateChosen_js\"(.*?)<\/script>/smi", '', $buffer);
		
		// Get special template scripts
		$scripts = VPFrameworkScript::getScripts();
		
		// Add scripts to buffer
		$buffer = preg_replace('#<vpf:include\ type="script"(.*)\/>#iU', $scripts, $buffer);
		
		// Optimize CSS, JS and HTML
		$buffer = VPFrameworkOptimizer::optimize($buffer, $this->params);

		// Set body buffer
		$app->setBody($buffer);
	}
	
	/**
	* Method to display special views
	* 
	* @param string $context
	* @param object $article
	* @param object $params
	* 
	* @return void
	*/
	public function displaySpecialViews($context, &$article, &$params)
	{
		if (false == in_array($context, array('com_content.article', 'com_content.category')))
		{
			return;
		}
		
		// If special views not enabled do nothing
		if (!$this->params->get('vm_wishlist', 1) && !$this->params->get('vm_compare', 1))
		{
			return;
		}
		
		$app                 = JFactory::getApplication();
		$template            = self::getTemplate();
		$compare_article_id  = (int) $this->params->get('compare_article_id', 0);
		$wishlist_article_id = (int) $this->params->get('wishlist_article_id', 1);

		if (!empty($compare_article_id) && $compare_article_id == $article->id)
		{
			$article->introtext = $template->renderComparePage();
			$article->text = $article->introtext;
			$article->fulltext = '';
			$article->readmore = 0;
			$article->images = '{"image_intro":"","float_intro":"","image_intro_alt":"","image_intro_caption":"","image_fulltext":"","float_fulltext":"","image_fulltext_alt":"","image_fulltext_caption":""}';
			$article->urls = '{"urla":"","urlatext":"","targeta":"","urlb":"","urlbtext":"","targetb":"","urlc":"","urlctext":"","targetc":""}';
			$article->params->set('isVMCompare', 1);
		}
		elseif (!empty($wishlist_article_id) && $wishlist_article_id == $article->id)
		{
			if (JFactory::getUser()->guest && !$this->params->get('allow_vm_guest_wishlist', 1))
			{
				$Itemid = $template->getArticleItemid($wishlist_article_id);
				$Itemid = !empty($Itemid) ? '&Itemid=' . $Itemid : '';
				$return = JRoute::_('index.php?option=com_content&view=article&id=' . $wishlist_article_id . $Itemid);
				$return = base64_encode(str_replace('&amp;', '&', $return));
				
				$app->enqueueMessage(JText::_('PLG_SYSTEM_VPFRAMEWORK_WISHLIST_VIEW_NEED_LOGIN'));
				$app->redirect(JRoute::_('index.php?option=com_users&view=login&return=' . $return, false));
				return;
			}
			
			$article->introtext = $template->renderWishlistPage();
			$article->text = $article->introtext;
			$article->fulltext = '';
			$article->readmore = 0;
			$article->images = '{"image_intro":"","float_intro":"","image_intro_alt":"","image_intro_caption":"","image_fulltext":"","float_fulltext":"","image_fulltext_alt":"","image_fulltext_caption":""}';
			$article->urls = '{"urla":"","urlatext":"","targeta":"","urlb":"","urlbtext":"","targetb":"","urlc":"","urlctext":"","targetc":""}';
			$article->params->set('isVMWishlist', 1);
		}
	}
	
	/**
	* Method to add additional parameters to admin form.
	* 
	* @param object $form  JForm object
	* @param array  $data  Form data
	* 
	* @return void
	*/
	public function addParamsToAdminForms(&$form, $data)
	{
		if (!$form)
		{
			return;
		}
		
		if (is_object($data))
		{
			$data = get_object_vars($data);
		}
		
		if (!empty($data['client_id']))
		{
			// Skip admin
			return;
		}
		
		$app     = JFactory::getApplication();
		$context = $form->getName();

		if ($context == 'com_menus.item') 
		{
			$formsToSkip = $this->getFormsToSkip('form');
			
			if (in_array('menu', $formsToSkip))
			{
				return;
			}
			
			JForm::addFormPath(JPATH_SITE . '/plugins/system/vpframework/forms');
			
			$form->loadFile('menu', false);
			
			$request = isset($data['request']) ? (array) $data['request'] : array();
			$option  = isset($request['option']) ? $request['option'] : '';
			$view    = isset($request['view']) ? $request['view'] : '';
			
			if ($option == 'com_users' && $view == 'login')
			{
				$form->loadFile('login', false);//vpdump($form);exit;
			}
		}
		elseif ($context == 'com_modules.module' || $context == 'com_config.modules' || $context == 'com_advancedmodules.module')
		{
			$formsToSkip = $this->getFormsToSkip('form');
			
			if (in_array('module', $formsToSkip))
			{
				return;
			}
			
			JForm::addFormPath(JPATH_SITE . '/plugins/system/vpframework/forms');
			
			$form->loadFile('module', false);
			
			$template = $this->getTemplate();

			if (empty($template))
			{
				return;
			}

			$template_form_path = JPATH_SITE . '/templates/' . $template->template . '/forms';
			
			if (file_exists($template_form_path))
			{
				JForm::addFormPath($template_form_path);
				
				$form->loadFile('module', false);
			}

		}
		elseif ($context == 'com_config.component')
		{
			if ($app->input->getCmd('component', '') == 'com_vmessentials')
			{
				$form->setFieldAttribute('auto_buttons', 'default', 0);
			}
		}
		elseif ($context == 'com_templates.style' || $context == 'com_config.templates')
		{
			$vpf_template_ids = $this->getTemplates('id');
			$template_id      = $this->input->getInt('id', 0);
			
			if (in_array($template_id, $vpf_template_ids))
			{
				$adminHelper = VPFrameworkAdmin::getInstance();
				$template = $adminHelper->getTemplate();
				
				if (empty($template))
				{
					return;
				}
				
				$formsToSkip = $this->getFormsToSkip('params');
				static $messageShown = false;
				
				if ($app->isClient('site') && !$messageShown)
				{
					$templateName = JText::_(strtoupper($template->template));
					$app->enqueueMessage('<br/><strong>VP Framework does not recommend front end editing of its template settings. ' .
					                     'Theme style customization does not work at front end.</strong><br/>' .
					                     'Login to Joomla! backend and go to Template Manager to edit ' . $templateName . ' template settings.', 'Warning');
					$messageShown = true;
				}
				
				jimport('joomla.filesystem.file');
				
				JForm::addFormPath(JPATH_SITE . '/plugins/system/vpframework/forms/params');
				
				$template_params_path = JPATH_SITE . '/templates/' . $template->template . '/forms/params';
				
				if (file_exists($template_params_path))
				{
					JForm::addFormPath($template_params_path);
				}
				
				$lessForm = JPath::clean(JPATH_SITE . '/templates/' . $template->template . '/less/less.xml');
				
				if ($app->isClient('administrator') && JFile::exists($lessForm) && !in_array('style', $formsToSkip))
				{
					$form->loadFile('style', false);
				}
				
				if (!in_array('navigation', $formsToSkip))
				{
					$form->loadFile('navigation', false);
				}
				
				if (VPFrameworkUtility::hasExtension('com_virtuemart', 'component') && !in_array('virtuemart', $formsToSkip))
				{
					$form->loadFile('virtuemart', false);
				}
				
				if (!in_array('optimization', $formsToSkip))
				{
					$form->loadFile('optimization', false);
				}
			}
		}
	}
	
	/**
	 * Called if user fails to be logged in.
	 *
	 * @param   array  $response  Array of response data.
	 *
	 * @return  void
	 */
	public function onUserLoginFailure($response)
	{
		$return = $this->input->post->get('return', '', 'BASE64');
		
		if ($this->tmpl == 'component' && !empty($return))
		{
			$app = JFactory::getApplication();
			
			if (is_array($response) && !empty($response['error_message']))
			{
				$app->enqueueMessage($response['error_message']);
			}
			
			$app->redirect(base64_decode($return));
			return;
		}
	}
	
	public function manageWishlist()
	{
		$user = JFactory::getUser();
		
		if (VPFrameworkUtility::hasExtension('com_virtuemart', 'component') && $this->params->get('allow_vm_guest_wishlist', 1) && !$user->guest)
		{
			VPFrameworkVM::moveWishlistToUser();
		}
	}
	
	public function mergeCustomFilterQuery(&$select, &$joinedTables, &$where, &$groupBy, &$orderBy, &$joinLang)
	{
		jimport( 'joomla.application.module.helper' );

		$module = JModuleHelper::getModule('mod_vp_custom_filter');

		if (empty($module->id))
		{
			return;
		}

		$modParams    = new JRegistry($module->params);
		$filterHelper = VPFrameworkFilter::getInstance($this->params, $modParams);

		return $filterHelper->mergeQuery($select, $joinedTables, $where, $groupBy, $orderBy, $joinLang);
	}
	
	/**
	* Method to get formatted bytes in MB, KB or butes based on given size
	* 
	* @param integer $mem_usage Size in bytes
	* 
	* @return string
	*/
	public static function calculateSize($mem_usage)
	{
		if ($mem_usage < 1024) 
			return $mem_usage . ' bytes'; 
		elseif ($mem_usage < 1048576) 
			return round($mem_usage/1024, 2) . ' KB'; 
		else 
			return round($mem_usage/1048576, 2) . ' MB';
	}
}