<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

class VPFrameworkAdmin extends JObject
{
	protected $_options;
	protected $input;
	protected $template;
	
	protected static $google_fonts = null;
	protected static $latest_versions = null;
	protected static $instance;
	
	public function __construct($options = array())
	{
		$this->_options = (array) $options;
		$app            = JFactory::getApplication();
		$this->input    = $app->input;
		$helper         = plgSystemVPFrameworkHelper::getInstance();
		$templates      = $helper->getTemplates();
		$id             = !empty($options['id']) ? $options['id'] : $app->input->getInt('id', 0);
		$this->template = false;
		
		if (empty($templates))
		{
			return false;
		}

		foreach ($templates as $template)
		{
			if ($template->id == $id)
			{
				$this->template = $template;
				break;
			}
		}
		
		return true;
	}
	
	public function getTemplate()
	{
		return $this->template;
	}
	
	public static function getInstance($options = array())
	{
		$options = (array) $options;
		$hash = md5(serialize($options));

		if (is_object(self::$instance))
		{
			return self::$instance;
		}

		self::$instance = new VPFrameworkAdmin($options);
		return self::$instance;
	}
	
	public function apply()
	{
		// Do not use this method in Joomla 4
		if (version_compare(JVERSION, '4.0.0', 'ge'))
		{
			return;
		}
		
		if (!JSession::checkToken())
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		jimport('joomla.html.parameter');
		jimport('legacy.model.legacy');
		jimport('legacy.model.form');
		jimport('legacy.model.admin');
		jimport('joomla.table.table');
		
		JTable::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_templates/tables');
		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_templates/models');
		JForm::addFormPath(JPATH_ADMINISTRATOR . '/components/com_templates/models/forms');
		defined('JPATH_COMPONENT') or define('JPATH_COMPONENT', JPATH_ADMINISTRATOR . '/components/com_templates');

		$app   = JFactory::getApplication();
		$lang  = JFactory::getLanguage();
		$model = JModelLegacy::getInstance('Style', 'TemplatesModel');
		$table = $model->getTable();
		$input = $app->input;
		$data  = $input->post->get('jform', array(), 'array');
		$checkin = property_exists($table, 'checked_out');
		$context = 'com_templates.edit.style';
		$key = $table->getKeyName();
		$recordId = $input->getInt($key);
		$listViewURL = JRoute::_('index.php?option=com_templates&view=styles', false);
		$thisViewURL = JRoute::_('index.php?option=com_templates&view=style&layout=edit&id=' . $recordId, false);

		// Populate the row id from the session.
		$data[$key] = $recordId;

		// Access check.
		if (!$this->allowSave($data, $key))
		{
			$this->returnFalse(JText::_('JLIB_APPLICATION_ERROR_SAVE_NOT_PERMITTED'), false, $listViewURL);
		}

		// Validate the posted data.
		// Sometimes the form needs some posted data, such as for plugins and modules.
		$form = $model->getForm($data, false);

		if (!$form)
		{
			$this->returnFalse($model->getError(), false);
		}

		// Test whether the data is valid.
		$validData = $model->validate($form, $data);

		// Check for validation errors.
		if ($validData === false)
		{
			$msgs = array();
			// Get the validation messages.
			$errors = $model->getErrors();

			// Push up to three validation messages out to the user.
			for ($i = 0, $n = count($errors); $i < $n && $i < 3; $i++)
			{
				if ($errors[$i] instanceof Exception)
				{
					$msgs[] = $errors[$i]->getMessage();
				}
				else
				{
					$msgs[] = $errors[$i];
				}
			}

			// Save the data in the session.
			$app->setUserState($context . '.data', $data);
			
			$this->returnFalse(implode('<br/>', $msgs), false);
		}

		if (!isset($validData['tags']))
		{
			$validData['tags'] = null;
		}

		// Attempt to save the data.
		if (!$model->save($validData))
		{
			// Save the data in the session.
			$app->setUserState($context . '.data', $validData);
			
			$msgs = array();
			$msgs[] = JText::sprintf('JLIB_APPLICATION_ERROR_SAVE_FAILED', $model->getError());
			$msgs[] = $this->getError();
			
			$this->returnFalse(implode('<br/>', $msgs), false);
		}

		// Save succeeded, so check-in the record.
		if ($checkin && $model->checkin($validData[$key]) === false)
		{
			// Save the data in the session.
			$app->setUserState($context . '.data', $validData);

			$msgs = array();
			$msgs[] = JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError());
			$msgs[] = $this->getError();
			
			$this->returnFalse(implode('<br/>', $msgs), false);
		}

		// Set the record data in the session.
		$recordId = $model->getState('style.id');
		$this->holdEditId($context, $recordId);
		$app->setUserState($context . '.data', null);
		$model->checkout($recordId);

		// Invoke the postSave method to allow for the child class to access the model.
		//$this->postSaveHook($model, $validData);

		$this->returnTrue(JText::_('JLIB_APPLICATION_SAVE_SUCCESS'));
	}

	/**
	 * Method to check if you can add a new record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array  $data  An array of input data.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowAdd($data = array())
	{
		$user = JFactory::getUser();
		return ($user->authorise('core.create', 'com_templates') || count($user->getAuthorisedCategories('com_templates', 'core.create')));
	}

	/**
	 * Method to check if you can edit an existing record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key; default is id.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowEdit($data = array(), $key = 'id')
	{
		return JFactory::getUser()->authorise('core.edit', 'com_templates');
	}

	/**
	 * Method to check if you can save a new or existing record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowSave($data, $key = 'id')
	{
		$recordId = isset($data[$key]) ? $data[$key] : '0';

		if ($recordId)
		{
			return $this->allowEdit($data, $key);
		}
		else
		{
			return $this->allowAdd($data);
		}
	}

	/**
	 * Method to add a record ID to the edit list.
	 *
	 * @param   string   $context  The context for the session storage.
	 * @param   integer  $id       The ID of the record to add to the edit list.
	 *
	 * @return  void
	 *
	 * @since   12.2
	 */
	protected function holdEditId($context, $id)
	{
		$app = JFactory::getApplication();
		$values = (array) $app->getUserState($context . '.id');

		// Add the id to the list if non-zero.
		if (!empty($id))
		{
			array_push($values, (int) $id);
			$values = array_unique($values);
			$app->setUserState($context . '.id', $values);

			if (defined('JDEBUG') && JDEBUG)
			{
				JLog::add(
					sprintf(
						'Holding edit ID %s.%s %s',
						$context,
						$id,
						str_replace("\n", ' ', print_r($values, 1))
					),
					JLog::INFO,
					'controller'
				);
			}
		}
	}
	
	public function updateGoogleFonts($ajax = true, $apiKey = null, $source = null)
	{
		if ($ajax && !JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.file');
		
		$app    = JFactory::getApplication();
		$apiKey = !empty($apiKey) ? $apiKey : 'AIzaSyAq2SR-XlurVcvRkEeZfchjuyaSLufblos';
		$source = !empty($source) ? $source : 'https://www.googleapis.com/webfonts/v1/webfonts';
		$url    = str_replace(' ', '%20', trim($source)) . '?key=' . $apiKey;
		$file   = JPath::clean(JPATH_SITE . '/media/vpframework/gfonts/fonts.json');
		$data   = @file_get_contents($url);
		
		if (empty($data))
		{
			if ($ajax)
			{
				$this->returnFalse('Google Fonts update failed.');
			}
			else
			{
				$app->enqueueMessage('Google Fonts update failed.');
				return false;
			}
		}
		
		if (!JFile::write($file, $data))
		{
			if ($ajax)
			{
				$this->returnFalse('Could not able to write with file ' . $file . '. Please makre sure the directory is not write protected.');
			}
			else
			{
				$app->enqueueMessage('Could not able to write with file ' . $file . '. Please makre sure the directory is not write protected.');
				return false;
			}
		}
		
		if ($ajax)
		{
			$this->returnTrue('Google Fonts updated from Google Server. Please wait as we refresh the page.', true);
		}
		else
		{
			$app->enqueueMessage('Fonts updated.');
			return true;
		}
	}
	
	public function getGoogleFonts($ajax = true)
	{
		if ($ajax && !JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		if (self::$google_fonts === null)
		{
			$app = JFactory::getApplication();
			$resource = JPath::clean(JPATH_SITE . '/media/vpframework/gfonts/fonts.json');
			
			if (!is_file($resource) || !file_exists($resource))
			{
				if (!$this->updateGoogleFonts($ajax) && !$ajax)
				{
					return false;
				}
			}
			
			$data = @file_get_contents($resource);
			
			if (empty($data))
			{
				if ($ajax)
				{
					$this->returnFalse('Failed to read Google Fonts from ' . str_replace(JPATH_SITE, '', $resource));
				}
				else
				{
					$app->enqueueMessage('Failed to read Google Fonts from ' . str_replace(JPATH_SITE, '', $resource));
					return false;
				}
			}
			
			$data = @json_decode($data, true);
			
			if (empty($data) || empty($data['items']))
			{
				if ($ajax)
				{
					$this->returnFalse('Failed to read Google Fonts from ' . str_replace(JPATH_SITE, '', $resource));
				}
				else
				{
					$app->enqueueMessage('Failed to read Google Fonts from ' . str_replace(JPATH_SITE, '', $resource));
					return false;
				}
			}
			
			self::$google_fonts = $data['items'];
		}
		
		if ($ajax)
		{
			$this->returnTrue(self::$google_fonts);
		}

		return self::$google_fonts;
	}
	
	public function getGoogleFont($font_name = null, $ajax = true)
	{
		if ($ajax && !JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		$app = JFactory::getApplication();
		$font_name = $ajax ? $app->input->getString('font_name', '') : $font_name;
		
		if (empty($font_name))
		{
			if ($ajax)
			{
				$this->returnFalse('Invalid font name');
			}
			else
			{
				$app->enqueueMessage('Invalid font name');
				return false;
			}
		}
		
		$fonts = $this->getGoogleFonts(false);
		
		if (empty($fonts))
		{
			if ($ajax)
			{
				$this->returnFalse($this->getMessages());
			}
			return false;
		}
		
		$font_name = urldecode($font_name);
		
		foreach ($fonts as $key => $font)
		{
			if ($font['family'] == $font_name)
			{
				if ($ajax)
				{
					$this->returnTrue($fonts[$key]);
				}
				else
				{
					return $fonts[$key];
				}
			}
		}

		if ($ajax)
		{
			$this->returnFalse('Unable to find font data for ' . $font_name);
		}

		return false;
	}
	
	public function buildHashes($ajax = true, $scanSection = null)
	{
		if ($ajax && !JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}

		if (empty($this->template))
		{
			return false;
		}
		
		$template_path       = JPATH_SITE . '/templates/' . $this->template->template;
		$media_path          = JPATH_SITE . '/media/vpframework';
		$template_hash_file  = $template_path . '/hashes.json';
		$framework_hash_file = VPF_PLUGINPATH . '/hashes.json';
		$template_folders    = array($template_path);
		$framework_folders   = array(VPF_PLUGINPATH, $media_path);
		
		if (empty($scanSection) || $scanSection == 'framework')
		{
			// Build has file for the framework
			if (!$this->_buildHashFiles($framework_folders, $framework_hash_file))
			{
				return false;
			}
		}

		if (empty($scanSection) || $scanSection == 'template')
		{
			// Build has file for the template
			if (!$this->_buildHashFiles($template_folders, $template_hash_file))
			{
				return false;
			}
		}
		
		return true;
	}
	
	protected function _buildHashFiles($foldersToScan, $hash_file)
	{
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.file');
		
		$hash               = array();
		$hash['SHA1Count']  = 0;
		$hash['SHA1']       = array();
		$filesToSkip        = array('index.html', 'CHANGELOG.txt', 'hashes.json', 'testreports.ini', '.htaccess');
		$filesToScan        = array();
		
		foreach ($foldersToScan as $folderToScan)
		{
			$files       = JFolder::files($folderToScan, '.', true, true);
			$filesToScan = !empty($files) ? array_merge($filesToScan, $files) : $filesToScan;
		}
		
		$hash['totalCount'] = count($filesToScan);
		
		foreach ($filesToScan as $fileToScan)
		{
			if (in_array(basename($fileToScan), $filesToSkip))
			{
				continue;
			}
			
			$file = (strpos($fileToScan, JPATH_SITE . '/') === 0) ? str_replace(JPATH_SITE . '/', '', $fileToScan) : str_replace(JPATH_SITE . '\\', '', $fileToScan);
			$file = str_replace('\\', '/', $file);
			
			$hash['SHA1'][$file] = sha1_file($fileToScan);
			
			$hash['SHA1Count']++;
		}
		
		$hash = json_encode($hash, JSON_PRETTY_PRINT);
		
		if (!JFile::write($hash_file, $hash))
		{
			return false;
		}
		
		return true;
	}

	public function getThemes()
	{
		if (empty($this->template))
		{
			return false;
		}
		
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.file');
		
		$themes_path = JPATH_SITE . '/templates/' . $this->template->template . '/css/themes';
		
		if (!JFolder::exists($themes_path))
		{
			return false;
		}
		
		$themes = JFolder::files($themes_path, '.css');
		
		if (!empty($themes))
		{
			foreach ($themes as $key => &$theme)
			{
				$themes[$key] = trim(preg_replace('/\\.[^.\\s]{3,4}$/', '', $theme));
			}
			
			return $themes;
		}
		
		return false;
	}
	
	public function getLayouts()
	{
		if (empty($this->template))
		{
			return false;
		}
		
		jimport('joomla.filesystem.folder');
		
		$layouts_path = JPATH_SITE . '/templates/' . $this->template->template . '/layouts';
		
		if (!JFolder::exists($layouts_path))
		{
			return false;
		}
		
		$layouts = JFolder::folders($layouts_path, '.');
		
		if (!empty($layouts))
		{
			foreach ($layouts as $key => &$layout)
			{
				$layouts[$key] = JFolder::makeSafe($layout);
			}
			return $layouts;
		}
		
		return false;
	}	

	public function getFrameworkVersion()
	{
		$xmlPath = JPath::clean(VPF_PLUGINPATH . '/vpframework.xml');
		
		if (version_compare(JVERSION, '3.0.0', 'ge'))
		{
			$xml = JFactory::getXML($xmlPath);
			$version = (string) $xml->version;
		}
		else
		{
			$parser = JFactory::getXMLParser('Simple');
			$parser->loadFile($xmlPath);
			$doc = $parser->document;
			$element = $doc->getElementByPath('version');
			$version = (string) $element->data();
		}
		
		return trim($version);
	}
	
	public function validateDlk()
	{
		JFactory::getCache('com_plugins', 'callback')->gc();
		
		return VPFrameworkDlk::getInstance()->validate();
	}
	
	public function revalidateDlk()
	{
		JFactory::getCache('com_plugins', 'callback')->gc();
		
		return VPFrameworkDlk::getInstance()->revalidate();
	}
	
	public function clearDlk()
	{
		JFactory::getCache('com_plugins', 'callback')->gc();
		
		return VPFrameworkDlk::getInstance()->clear();
	}
	
	// Sample request index.php?option=com_templates&view=style&layout=edit&id={template_style_id}&task=style.findUpdate&eid={extension_id}&installed_version={installed_version}&{token}=1
	
	public function findUpdate()
	{
		if (!JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		$installed_version = $this->input->getCmd('installed_version', '');
		$update            = $this->findUpdates();
		
		$result  = array('updateFound' => false, 'version' => $installed_version, 'infourl' => '');
		
		if (empty($update))
		{
			$this->returnTrue($result);
		}

		if (!empty($installed_version) && version_compare($update->version, $installed_version, '>'))
		{
			$result['updateFound'] = true;
			$result['version'] = $update->version;
			$result['infourl'] = $update->infourl;
		}
		else
		{
			$result['updateFound'] = false;
			$result['version'] = $installed_version;
		}
		
		$this->returnTrue($result);
	}
	
	public function findUpdates()
	{
		$app               = JFactory::getApplication();

		$eid               = $this->input->getInt('eid', 0);
		$eid               = empty($eid) ? $this->input->getInt('extension_id', 0) : $eid;
		$skip              = $this->input->get('skip', array(), 'array');
		$cache_timeout     = $this->input->getInt('cache_timeout', 0);
		$minimum_stability = $this->input->getInt('minimum_stability', -1);

		$component         = JComponentHelper::getComponent('com_installer');
		$params            = $component->params;

		if ($cache_timeout == 0)
		{
			$cache_timeout = $params->get('cachetimeout', 6, 'int');
			$cache_timeout = 3600 * $cache_timeout;
		}

		if ($minimum_stability < 0)
		{
			$minimum_stability = $params->get('minimum_stability', JUpdater::STABILITY_STABLE, 'int');
		}

		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_installer/models');
		/** @var InstallerModelUpdate $model */
		$model = JModelLegacy::getInstance('Update', 'InstallerModel', array('ignore_requests' => true));
		$model->findUpdates($eid, $cache_timeout, $minimum_stability);

		$model->setState('list.start', 0);
		$model->setState('list.limit', 0);

		if ($eid != 0)
		{
			$model->setState('filter.extension_id', $eid);
		}

		$updates = $model->getItems();
		
		if (!empty($eid))
		{
			foreach ($updates as $update)
			{
				if ($update->extension_id == $eid)
				{
					return $update;
				}
			}
			
			return false;
		}
		
		if (!empty($skip))
		{
			$unfiltered_updates = $updates;
			$updates            = array();

			foreach ($unfiltered_updates as $update)
			{
				if (!in_array($update->extension_id, $skip))
				{
					$updates[] = $update;
				}
			}
		}

		return $updates;
	}

	public function getFilesIntegrity($ajax = true, $scanSection = null)
	{
		$app = JFactory::getApplication();
		
		if ($ajax && !JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		if (empty($this->template))
		{
			if ($ajax)
			{
				$this->returnFalse('Template information not found', false, 'index.php');
			}
			else
			{
				$app->enqueueMessage('Template information not found', 'warning');
				return false;
			}
		}
		
		$framework_result = array();
		$template_result  = array();
		$final_result     = array();
		
		if (empty($scanSection) || $scanSection == 'framework')
		{
			$media_path    = JPATH_SITE . '/media/vpframework';
			$foldersToScan = array(VPF_PLUGINPATH, $media_path);
			$hash_file     = VPF_PLUGINPATH . '/hashes.json';
			
			if (!$framework_result = $this->_compareHashes($foldersToScan, $hash_file))
			{
				if ($ajax)
				{
					$this->returnFalse($this->getError());
				}
				else
				{
					$app->enqueueMessage($this->getError(), 'warning');
					return false;
				}
			}
		}
		
		if (empty($scanSection) || $scanSection == 'template')
		{
			$template_path = JPATH_SITE . '/templates/' . $this->template->template;
			$foldersToScan = array($template_path);
			$hash_file     = $template_path . '/hashes.json';
			
			if (!$template_result = $this->_compareHashes($foldersToScan, $hash_file))
			{
				if ($ajax)
				{
					$this->returnFalse($this->getError());
				}
				else
				{
					$app->enqueueMessage($this->getError(), 'warning');
					return false;
				}
			}
		}
		
		if (empty($framework_result))
		{
			$final_result = $template_result;
		}
		elseif (empty($template_result))
		{
			$final_result = $framework_result;
		}
		elseif (!empty($framework_result) && !empty($template_result))
		{
			foreach ($framework_result as $key => $value)
			{
				if (!empty($template_result[$key]))
				{
					if (is_array($value) && is_array($template_result[$key]))
					{
						$final_result[$key] = array_merge($value, $template_result[$key]);
					}
					else
					{
						$final_result[$key] = $value + $template_result[$key];
					}
				}
				else
				{
					$final_result[$key] = $value;
				}
			}
		}
		
		if ($ajax)
		{
			$final_result['changedFiles'] = !empty($final_result['changedFiles']) ? implode(PHP_EOL, $final_result['changedFiles']) : '';
			$final_result['deletedFiles'] = !empty($final_result['deletedFiles']) ? implode(PHP_EOL, $final_result['deletedFiles']) : '';
			$final_result['newFiles']     = !empty($final_result['newFiles']) ? implode(PHP_EOL, $final_result['newFiles']) : '';
			
			$this->returnTrue($final_result);
		}
		else
		{
			return $final_result;
		}
	}
	
	protected function _compareHashes($foldersToScan, $hash_file)
	{
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.file');
		
		if (!JFile::exists($hash_file))
		{
			$this->setError('Hash file not found. File - ' . $hash_file);
			return false;
		}
		
		$hash = @file_get_contents($hash_file);
		$hash = !empty($hash) ? @json_decode($hash, true) : null;
		
		if (empty($hash))
		{
			$this->setError('Failed to read hashes. File - ' . $hash_file);
			return false;
		}
		
		$hash_totalCount = !empty($hash['totalCount']) ? $hash['totalCount'] : 0;
		$hash_SHA1Count  = !empty($hash['SHA1Count']) ? $hash['SHA1Count'] : 0;
		$hash_SHA1       = !empty($hash['SHA1']) ? $hash['SHA1'] : array();
		$filesToScan     = array();
		
		foreach ($foldersToScan as $folderToScan)
		{
			$files       = JFolder::files($folderToScan, '.', true, true);
			$filesToScan = !empty($files) ? array_merge($filesToScan, $files) : $filesToScan;
		}
		
		$totalCount             = count($filesToScan);
		$SHA1Count              = 0;
		$SHA1                   = array();
		$result                 = array();
		$result['newFiles']     = array();
		$result['changedFiles'] = array();
		$filesToSkip            = array('index.html', 'CHANGELOG.txt', 'hashes.json', 'testreports.ini', '.htaccess');
		
		foreach ($filesToScan as $fileToScan)
		{
			if (in_array(basename($fileToScan), $filesToSkip))
			{
			 continue;
			}
			
			$file = (strpos($fileToScan, JPATH_SITE . '/') === 0) ? str_replace(JPATH_SITE . '/', '', $fileToScan) : str_replace(JPATH_SITE . '\\', '', $fileToScan);
			$file = str_replace('\\', '/', $file);
			$file_SHA1 = sha1_file($fileToScan);
			
			if (array_key_exists($file, $hash_SHA1))
			{
				if ($hash_SHA1[$file] != $file_SHA1)
				{
					$result['changedFiles'][] = $file;
				}
			}
			else
			{
				$result['newFiles'][] = $file;
			}
			
			$SHA1[$file] = $file_SHA1;
		}
		
		$result['deletedFiles'] = array_diff_key($hash_SHA1, $SHA1);
		$result['deletedFiles'] = !empty($result['deletedFiles']) ? array_keys($result['deletedFiles']) : $result['deletedFiles'];
		$result['newCount']     = count($result['newFiles']);
		$result['changedCount'] = count($result['changedFiles']);
		$result['deletedCount'] = count($result['deletedFiles']);
		
		return $result;
	}
	
	public function getLessVars($less_file = null)
	{
		$less_file = !empty($less_file) ? $less_file : JPATH_SITE . '/templates/' . $this->template->template . '/css/themes/default.less';
		
		if (!is_file($less_file))
		{
			return false;
		}

		$less_vars = file_get_contents($less_file);
		
		if (!empty($less_vars))
		{
			// Remove comments
			$less_vars = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $less_vars);
			$less_vars = explode(';', $less_vars);
			
			if (!empty($less_vars))
			{
				$array = array();

				foreach ($less_vars as $var)
				{
					if (strpos($var, ':') !== false)
					{
						list($key, $value) = explode(':', $var);
						$key = trim(str_replace('@', '', $key));
						$array[$key] = (string) trim($value);
					}
				}
				
				return $array;
			}
		}
		
		return false;
	}
	
	public function saveLessVars($less_file, $less_vars)
	{
		jimport ('joomla.filesystem.file');
		
		if (!empty($less_vars) && is_array($less_vars))
		{
			$content = '';
			
			foreach ($less_vars as $name => $value)
			{
				$content .= '@' . $name . ': ' . $value . ';' . PHP_EOL;
			}
			
			if (!JFile::write($less_file, $content))
			{
				return false;
			}
			
			return true;
		}
		return false;
	}
	
	public function getLessForm($selectedTheme = 'default', $reloadVars = false)
	{
		jimport ('joomla.filesystem.file');
		$lessFormPath  = JPath::clean(JPATH_SITE . '/templates/' . $this->template->template . '/less');
		
		if (!JFile::exists ($lessFormPath . '/less.xml'))
		{
			return false;
		}
		
		JForm::addFormPath($lessFormPath);
		$form = JForm::getInstance('tpl_' . $this->template->template . '.less', 'less', array('control' => 'jform'));
		
		if ($selectedTheme !== false && !empty($form))
		{
			$selectedTheme = !empty($selectedTheme) ? $selectedTheme : 'default';
			$lessVarFile   = JPath::clean(JPATH_SITE . '/templates/' . $this->template->template . '/css/themes/' . $selectedTheme . '.less');
			$context       = 'com_templates.style.tpl_' . $this->template->template;
			$userStateVars = JFactory::getApplication()->getUserState($context . '.theme_data', null);
			$lessVars      = ($userStateVars !== null && !$reloadVars) ? $userStateVars : $this->getLessVars($lessVarFile);
			
			if (!empty($lessVars))
			{
				$data = new stdClass;
				$data->less = $lessVars;
				$form->bind($data);
				
				foreach ($form->getFieldsets() as $key => $fieldset) 
				{
					$fields = $form->getFieldset($fieldset->name);
					
					if (count($fields))
					{
						foreach ($fields as $field)
						{
							if (strtolower($field->type) === 'media' && $field->getAttribute('lessformat') === 'imgurl')
							{
								$name   = $field->getAttribute('name');
								$oValue = $field->value;
								$value  = '';
								
								if (!empty($oValue) && strpos($oValue, 'url(') !== false)
								{
									$value = str_replace(array("url('../../../../", "')"), array("", ""), $oValue);
								}

								$form->setValue($name, 'less', $value);
							}
						}
					}
				}
			}
		}
		
		return $form;
	}
	
	public function validateForm($form, $data, $group = null)
	{
		// Filter and validate the form data.
		$data = $form->filter($data);
		$return = $form->validate($data, $group);

		// Check for an error.
		if ($return instanceof Exception)
		{
			$this->setError($return->getMessage());

			return false;
		}

		// Check the validation results.
		if ($return === false)
		{
			// Get the validation messages from the form.
			foreach ($form->getErrors() as $message)
			{
				$this->setError($message);
			}

			return false;
		}

		return $data;
	}
	
	public function getLessFields($selectedTheme = 'default', $reloadVars = false)
	{
		$form = $this->getLessForm($selectedTheme, $reloadVars);
		$html = '';
		$needClosing = false;
		$i = 0;
		
		if (!empty($form))
		{
			$html .= '<fieldset class="form-horizontal">';
			$html .= '<legend>Customize Theme</legend>';
			
			foreach ($form->getFieldsets() as $key => $fieldset) 
			{
				$fields = $form->getFieldset($fieldset->name);
				if (count($fields))
				{
					$html .= '<div class="less-fieldset' . (($i == 0) ? ' active' : '') . '">';
					if (!empty($fieldset->label))
					{
						$html .= '<div class="control-group">';
						$html .= '<h4 class="vpf-group-heading" onclick="return VPFramework.displayLessGroup(this);"><span>' . $fieldset->label . '</span><span class="opener"></span></h4>';
						if (!empty($fieldset->labelnote))
						{
							$html .= '<div class="vpf-group-heading-desc"><span class="small muted">' . $fieldset->labelnote . '</span></div>';
						}
						$html .= '</div>';
						$html .= '<div class="less-group-content">';
						$needClosing = true;
					}
					foreach ($fields as $field)
					{
						if ($field->hidden)
						{
							$html .= $field->input;
						}
						else
						{
							$html .= '<div class="control-group">';
							$html .= '<div class="control-label">';
							$html .= $field->label;
							if ($field->getAttribute('labelnote'))
							{
								$html .= '<div class="small muted label-note">' . $field->getAttribute('labelnote') . '</div>';
							}
							$html .= '</div>';
							$html .= '<div class="controls">';
							$html .= $field->input;
							if ($field->getAttribute('note'))
							{
								$html .= '<div class="small muted">' . $field->getAttribute('note') . '</div>';
							}
							$html .= '</div>';
							$html .= '</div>';
						}
					}
					if ($needClosing)
					{
						$html .= '</div>';
						$needClosing = false;
					}
					$html .= '</div>';
				}
				$i++;
			}
			$html .= '<div class="loading hide"></div>';
			$html .= '</fieldset>';
		}

		return $html;
	}
	
	public function getLESSFieldsAjax()
	{
		if (!JSession::checkToken('GET'))
		{
			$this->returnFalse(JText::_('JINVALID_TOKEN'), false, 'index.php');
		}
		
		if (empty($this->template))
		{
			$this->returnFalse('Template information not found', false, 'index.php');
		}
		
		$selectedTheme = $this->input->getCmd('theme', 'default');
		$fields = $this->getLessFields($selectedTheme, true);
		$this->returnTrue($fields);
	}
	
	public function compileCSS($less_file, $vars = null)
	{
		if (!class_exists('lessc')) require VPF_PLUGINPATH . '/lib/lessc.inc.php';
		
		$vars = (array) !empty($vars) ? $vars : $this->getLessVars();
		$less = new lessc;
		$less->setPreserveComments(true);
		
		$less->registerFunction('addBorder', function($arg)
		{
			if (isset($arg[2]))
			{
				$input = $arg[2];
				$value = isset($input[0]) && isset($input[0][1]) ? $input[0][1] : 'transparent';
				$width = isset($input[1]) && isset($input[1][1]) ? $input[1][1] : '1';
				$unit = isset($input[1]) && isset($input[1][2]) ? $input[1][2] : 'px';
				
				if ($value == 'transparent' || $value == 'none' || $value == '')
				{
					return '0 none';
				}
				else
				{
					return $width . $unit . ' solid ' . $value;
				}
			}
			
			return '0 none';
		});
		
		if (!is_file($less_file) || !file_exists($less_file))
		{
			return false;
		}

		if (!empty($vars))
		{
			$less->setVariables($vars);
		}
		
		return $less->compileFile($less_file);
	}
	
	public function saveTheme($theme, $data, $newTheme, $table)
	{
		jimport ('joomla.filesystem.file');
		
		$app     = JFactory::getApplication();
		$themes  = $this->getThemes();
		$context = 'com_templates.style.tpl_' . $this->template->template;
		
		$app->setUserState($context . '.theme_data', $data);
		
		if ($newTheme !== false)
		{
			$app->setUserState($context . '.new_theme', $newTheme);
			
			if (empty($newTheme))
			{
				$table->setError('Theme name was missing. New theme creation failed.');
				return false;
			}
			
			$newTheme = JFile::makeSafe(strtolower(str_replace(' ', '_', $newTheme)));
			$app->setUserState($context . '.new_theme', $newTheme);
		}
		
		if (!empty($newTheme) && in_array($newTheme, $themes))
		{
			$table->setError('Theme name already exists. Please use a new name.');
			return false;
		}
		
		// Validate LESS form data
		$form = $this->getLessForm(false);
		$formData = array();
		$formData['less'] = $data;
		$validData = $this->validateForm($form, $formData);
		
		// Check for validation errors.
		if ($validData === false)
		{
			// Get the validation messages.
			$errors = $this->getErrors();
			$messages = array();

			// Push up to three validation messages out to the user.
			for ($i = 0, $n = count($errors); $i < $n && $i < 3; $i++)
			{
				if ($errors[$i] instanceof Exception)
				{
					$messages[] = $errors[$i]->getMessage();
				}
				else
				{
					$messages[] = $errors[$i];
				}
			}

			// Save the data in the session.
			$app->setUserState($context . '.theme_data', $data);
			
			if (!empty($messages))
			{
				$messages = implode('<br/>', $messages);
				$table->setError($messages);
			}

			return false;
		}
		
		$this->formatLessData($form, $validData);
		
		// Replace raw data by validated data
		$data = $validData['less'];
		
		$theme = !empty($newTheme) ? $newTheme : $theme;
		$lessFile = JPath::clean(JPATH_SITE . '/templates/' . $this->template->template . '/less/theme.less');
		$themeLessVarFile = JPath::clean(JPATH_SITE . '/templates/' . $this->template->template . '/css/themes/' . $theme . '.less');
		$themeCSSFile = JPath::clean(JPATH_SITE . '/templates/' . $this->template->template . '/css/themes/' . $theme . '.css');
		
		if (!JFile::exists($lessFile))
		{
			$table->setError('Unable to create new theme. Theme LESS file is missing.');
			return false;
		}
		
		if (!$this->saveLessVars($themeLessVarFile, $data))
		{
			$table->setError('Failed to save LESS variables. Please make sure "/templates/' . $this->template->template . '/css/themes" is not write protected.');
			return false;
		}
		
		try
		{
			$css = $this->compileCSS($lessFile, $data);
		}
		catch (Exception $e)
		{
			$table->setError($e->getMessage());
			return false;
		}
		
		if (!JFile::write($themeCSSFile, $css))
		{
			$table->setError('Unable to save CSS file. Please make sure "/templates/' . $this->template->template . '/css/themes" is not write protected.');
			return false;
		}
		
		// Theme save. Clear user state.
		$app->setUserState($context . '.new_theme', null);
		$app->setUserState($context . '.theme_data', null);
		
		// Set the new theme in template params
		$params = $table->get('params');
		$registry = new JRegistry;
		$registry->loadString($params);
		$registry->set('theme', $theme);
		$params = $registry->toString();
		$table->set('params', $params);
		
		// Clean cache
		$this->cleanCache('vp_framework');
		$this->cleanCache('vp_framework_assets');
		
		return true;
	}
	
	protected function formatLessData($form, &$validData)
	{
		foreach ($form->getFieldsets() as $set => $fieldset)
		{
			$fields = $form->getFieldset($set);
			
			if (empty($fields)) continue;
			
			foreach ($fields as $field)
			{
				$format = $form->getFieldAttribute($field->fieldname, 'lessformat', null, 'less');
				
				if ($format && !empty($validData['less']) && isset($validData['less'][$field->fieldname]))
				{
					if (strtolower($format) == 'imgurl')
					{
						$url = $validData['less'][$field->fieldname];
						
						if (empty($url))
						{
							$validData['less'][$field->fieldname] = 'none';
						}
						elseif (strpos($url, 'url(') === false && $url != 'none')
						{
							if (strpos($url, '#') !== false)
							{
								$parts = explode('#', $url);
								$url   = $parts[0];
							}
							
							$validData['less'][$field->fieldname] = "url('../../../../" . $url . "')";
						}
					}
				}
			}
		}
	}
	
	protected function cleanCache($group, $client_id = 0)
	{
		$conf = JFactory::getConfig();

		$options = array(
			'defaultgroup' => $group,
			'cachebase' => ($client_id) ? JPATH_ADMINISTRATOR . '/cache' : $conf->get('cache_path', JPATH_SITE . '/cache')
		);

		$cache = JCache::getInstance('callback', $options);
		$cache->clean();

		if (version_compare(JVERSION, '4.0.0', 'ge'))
		{
			JFactory::getApplication()->triggerEvent('onExtensionCleanCache', $options);
		}
		else
		{
			// Trigger the onContentCleanCache event.
			JEventDispatcher::getInstance()->trigger('onExtensionCleanCache', $options);
		}
	}

	private function getMessages()
	{
		return VPFrameworkUtility::getMessages();
	}
		
	protected function returnTrue($msg = '', $reload = false, $redirect = null)
	{
		$output = array();
		$output['error'] = false;
		$output['msg'] = $msg;
		$output['reload'] = $reload;
		$output['redirect'] = $redirect;
		
		return $this->jsonReturn($output);
	}

	protected function returnFalse($msg = '', $reload = false, $redirect = null)
	{
		$output = array();
		$output['error'] = true;
		$output['msg'] = $msg;
		$output['reload'] = $reload;
		$output['redirect'] = $redirect;
		
		return $this->jsonReturn($output);
	}
	
	protected function jsonReturn($output = array()) 
	{
		return VPFrameworkUtility::jsonReturn($output);
	}
}
