<?php
/**
 *
 * @package 	customfilters
 * @author		Sakis Terz
 * @copyright	Copyright (c) 2012-2022 breakdesigns.net. All rights reserved.
 * @license		http://www.gnu.org/copyleft/gpl.html GNU/GPL, see LICENSE.php
 */

defined('_JEXEC') or die();

use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Http\HttpFactory;
use Joomla\Registry\Registry;
use Joomla\CMS\Client\ClientHelper;
use Joomla\CMS\Client\FtpClient;
use Joomla\CMS\Filesystem\Path;

/**
 * Used for connecting with the Breakdesigns site and checks for the latest version's info
 *
 *
 */
class extensionUpdateHelper
{

    public static $instance;

    private $updateDataFile;

    private $targetFileExists;

    private $extension;

    /* The update frequency in hours */
    private $updateFrequency;

    /**
     * Implements the Singleton pattern for this clas
     *
     * @staticvar extensionUpdateHelper $instance The static object instance
     * @return extensionUpdateHelper
     */
    public static function getInstance($extension = null, $target = null, $updateFrequency = null)
    {
        if (! isset(self::$instance)) {
            self::$instance = new extensionUpdateHelper($extension, $target, $updateFrequency);
        }
        return self::$instance;
    }

    public function __construct($extension = null, $target = null, $updateFrequency = 2)
    {
        if (! isset($extension) || ! isset($target)) {
            throw new Exception('No extension or target defined');
        }
        $this->extension = $extension;
        $this->updateFrequency = $updateFrequency;

        if (File::exists(JPATH_COMPONENT_ADMINISTRATOR . DIRECTORY_SEPARATOR . $target)) {
            $this->targetFileExists = true;
        } else {
            $this->targetFileExists = false;
        }
        $this->updateDataFile = JPATH_COMPONENT_ADMINISTRATOR . DIRECTORY_SEPARATOR . $target;
    }

    /**
     * Main function that triggers the processes
     *
     * @return boolean|stdClass
     */
    public function getData()
    {
        $file_created = true;
        $data = '';
        if ($this->existFOPEN()) {
            if ($this->targetFileExists) {} else {
                $file_created = $this->createFile();
            }
        }

        // if the file does not exist and cannot be created get the data directly from the url
        if ($file_created === false) {
            $data = $this->downloadData();
        } else
            if (@ filesize($this->updateDataFile) > 0) {
                // decide if should read from contents or a download should be done
                $file_modif_time = filemtime($this->updateDataFile);
                $updateFrequency_in_sec = (int) $this->updateFrequency * 60;
                $current_timestamp = time();
                // if the frequency time has passed download
                if ($file_modif_time === false || $current_timestamp - $file_modif_time > $updateFrequency_in_sec) {
                    $data = $this->downloadData();
                    File::write($this->updateDataFile, $data);
                } elseif(File::exists($this->updateDataFile)) {
                    $data = \file_get_contents($this->updateDataFile);
                }
            } else {
                // file exists but its empty,download the data and write them to the file
                $data = $this->downloadData();
                File::write($this->updateDataFile, $data);
            }
        if (! empty($data)) {
            $registry = new Registry('update');
            $registry->loadString($data, 'INI');
            return $registry->toObject();
        } else {
            return false;
        }
    }

    /**
     * Downloads the data from a specified URL
     *
     * @param string $url
     * @return string $data;
     */
    private function downloadData()
    {
        if ($this->extension) {
            $url = 'https://breakdesigns.net/index.php?option=com_extensiondata&format=ini&extension=' . $this->extension;
        }
        $http = HttpFactory::getHttp();
        $result = $http->get($url);
        return $result->body;
    }

    /**
     * Create the file if it does no exist
     *
     * @param string $file
     *            path
     */
    private function createFile()
    {
        $target = $this->updateDataFile;
        $fp = @fopen($target, 'wb');
        if ($fp === false) {
            // The file can not be opened for writing. Let's try a hack.
            if (File::write($target, '')) {
                if (self::chmod($target, 511)) {
                    $fp = @fopen($target, 'wb');
                }
            }
        }
        return $fp;
    }

    /**
     * Does the server support URL fopen() wrappers?
     *
     * @return bool
     */
    private function existFOPEN()
    {
        // If we are not allowed to use ini_get, we assume that URL fopen is disabled
        if (! function_exists('ini_get')) {
            $result = false;
        } else {
            $result = ini_get('allow_url_fopen');
        }
        return $result;
    }

    /**
     * Does the server support PHP's cURL extension?
     *
     * @return bool True if it is supported
     */
    private function existCURL()
    {
        $result = function_exists('curl_init');
        return $result;
    }

    /**
     * Change the permissions of a file, optionally using FTP
     *
     * @param string $file
     *            Absolute path to file
     * @param int $mode
     *            Permissions, e.g. 0755
     * @copyright Copyright (c)2012-2018 Nicholas K. Dionysopoulos / Sakis Terzis
     */
    private static function chmod($path, $mode)
    {
        if (is_string($mode)) {
            $mode = octdec($mode);
            if (($mode < 0600) || ($mode > 0777)) {
                $mode = 0755;
            }
        }

        $ftpOptions = ClientHelper::getCredentials('ftp');

        // Check to make sure the path valid and clean
        $path = Path::clean($path);

        if ($ftpOptions['enabled'] == 1) {
            // Connect the FTP client
            $ftp = FtpClient::getInstance($ftpOptions['host'], $ftpOptions['port'], null, $ftpOptions['user'], $ftpOptions['pass']);
        }

        if (@chmod($path, $mode)) {
            $ret = true;
        } elseif ($ftpOptions['enabled'] == 1) {
            // Translate path and delete
            $path = Path::clean(str_replace(JPATH_ROOT, $ftpOptions['root'], $path), '/');
            // FTP connector throws an error
            $ret = $ftp->chmod($path, $mode);
        } else {
            $ret = false;
        }
        return $ret;
    }

    /**
     *
     * Convert a plain text to array based on line breaks
     *
     * @param string $text
     * @return array
     */
    public static function convertTextToArray($text)
    {
        $lines = explode("\n", $text);
        foreach ($lines as $line) {
            $pos = strpos($line, '=');
            if ($pos !== false) {
                $line = str_replace('"', '', $line);
                $key = substr($line, 0, strlen($line) - $pos);
                $value = substr($line, $pos + 1);
                echo $key, '--' . $value;
            }
        }
        return;
    }
}
