<?php
/**
 * Installation file for CustomFilters
 *
 * @package customfilters.install
 * @author Sakis Terzis
 * @copyright Copyright (c) 2012-2022 breakdesigns. All rights reserved.
 * @license  GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Application\ApplicationHelper;

/**
 * Load the installer
 * @access        public
 */
class pkg_customfiltersInstallerScript
{
    /**
     * The name of our package, e.g. pkg_example. Used for dependency tracking.
     *
     * @var  string
     */
    protected $packageName = 'pkg_customfilters.xml';

    /**
     * The name of our component, e.g. com_example. Used for dependency tracking.
     *
     * @var  string
     */
    protected $componentName = 'com_customfilters';

    /**
     * The minimum PHP version required to install this extension
     *
     * @var   string
     */
    protected $minimumPHPVersion = '5.4.0';

    /**
     * The minimum Joomla! version required to install this extension
     *
     * @var   string
     */
    protected $minimumJoomlaVersion = '3.8.1';

    /**
     * The maximum Joomla! version this extension can be installed on
     *
     * @var   string
     */
    protected $maximumJoomlaVersion = '5.0.0';

    /**
     * @var array
     * @since 2.15.0
     */
    protected $databaseExtension = [];

    /**
     * Set defaults on update.
     * Usage: We may do not want to break existing features (e.g. urls) on update.
     * But offer SEF urls on new installations (default setting)
     *
     * @var []
     * @since 2.15.0
     */
    protected $componentDefaultSettings = [
            'sef_urls_for_cf' => 0
    ];

    /**
     * A list of extensions (modules, plugins) to enable after installation. Each item has four values, in this order:
     * type (plugin, module, ...), name (of the extension), client (0=site, 1=admin), group (for plugins).
     *
     * @var array
     */
    protected $extensionsToEnable = array(
        array('plugin', 'breakdesignsajax', 1, 'system'),
    );

    /**
     * @var Joomla\Registry\Registry
     * @since 2.15.0
     */
    protected $componentParams;

    /**
     * Update messages, contain critical features added in each version
     *
     * @var array
     */
    protected $messages = [
        '2.2.2' => "New feature: The min and max price range can be calculated dynamically for a set of products",
        '2.2.10' => "New feature: Can search/filter the child products and return as results the parent products",
        '2.7.0' => "New feature: A new module for displaying the selected filters, as breadcrumbs/tags",
        '2.10.0' => "New feature: Set loading dependencies between the custom filters.",
        '2.15.0' => "New feature: SEF URLs also for the Custom Filters.</br>Check the configuration settings in the 'VM Custom Filters PRO' component."
    ];

    /**
     * @var stdClass
     */
    protected $extensionsStatus;

    /**
     * Update routine
     *
     * @param
     * @return
     * @author        Sakis Terzis
     * @access        public
     * @since        2.0
     */
    public function update($parent)
    {

    }

    /**
     * Preflight routine executed before install and update
     *
     * @param        $type    string    type of change (install, update or discover_install)
     * @return
     * @todo
     * @see
     * @access        public
     * @copyright
     * @author        Sakis Terzis
     * @since        2.0
     */
    public function preflight($type, $parent)
    {
        // Check the minimum PHP version
        if (!version_compare(PHP_VERSION, $this->minimumPHPVersion, 'ge')) {
            $msg = "<p>You need PHP $this->minimumPHPVersion or later to install this package</p>";
            Log::add($msg, Log::WARNING, 'jerror');

            return false;
        }

        // Check the minimum Joomla! version
        if (!version_compare(JVERSION, $this->minimumJoomlaVersion, 'ge')) {
            $msg = "<p>You need Joomla! $this->minimumJoomlaVersion or later to install this component</p>";
            Log::add($msg, Log::WARNING, 'jerror');

            return false;
        }

        // Check the maximum Joomla! version
        if (!version_compare(JVERSION, $this->maximumJoomlaVersion, 'le')) {
            $msg = "<p>You need Joomla! $this->maximumJoomlaVersion or earlier to install this component</p>";
            Log::add($msg, Log::WARNING, 'jerror');

            return false;
        }

        if ($type == 'update') {
            $milestone_versions = array_keys($this->messages);
            $this->printed_messages = array();
            $oldRelease = $this->getParam('version');
            foreach ($milestone_versions as $m_v) {
                if (version_compare($oldRelease, $m_v) == -1) {
                    $this->printed_messages[] = $this->messages[$m_v];
                }
            }

            // We set the params in the preflight, so that we can use the old version var (manifest)
            $this->setComponentDefaultParams();
        }

        //delete the latest version ini , to create a new one for the updated version
        $version_ini_path = JPATH_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_customfilters' . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'lastversion.ini';
        if (File::exists($version_ini_path)) {
            File::delete($version_ini_path);
        }
    }

    /**
     * Get the files included in the package's xml under the files node
     *
     * @param $manifest
     * @return array
     */
    private function getPackageExtensions($manifest)
    {
        $includedFiles=[];
        $files=$manifest->xpath('files/file');
        foreach ($files as $file){
            $type=(string)$file->attributes()->type;
            $name=(string)$file->attributes()->id;
            $description=(string)$file->attributes()->description;
            $includedFiles[]=['name'=>$name, 'type'=>$type, 'description'=>$description];
        }
        return $includedFiles;
    }

    /**
     * Get a variable from the manifest file (actually, from the manifest cache).
     *
     * @param string $name
     *
     * @return string
     * @since 2.0.0
     */
    private function getParam(string $name)
    {
        $param = '';
        $extension = $this->getInstalledExtension();
        if($extension) {
            $manifest = new Joomla\Registry\Registry($extension->manifest_cache);
            $param = $manifest->get($name);
        }
        return $param;
    }

    /**
     * Get an extension from the db
     *
     * @param   string  $extensionName
     *
     * @return array
     * @since 2.15.0
     */
    protected function getInstalledExtension($extensionName = "com_customfilters")
    {
        if(!isset($this->databaseExtension[$extensionName])) {
            /** @var DatabaseDriver $db */
            $db = Factory::getDbo();
            $query = $db->getQuery(true);
            $query->select([$db->quoteName('client_id'), $db->quoteName('manifest_cache'), $db->quoteName('params')])
                  ->from($db->quoteName('#__extensions'))
                  ->where($db->quoteName('element') . '=' . $db->quote($extensionName));
            $db->setQuery($query);
            $this->databaseExtension[$extensionName] = $db->loadObject();
        }
        return $this->databaseExtension[$extensionName];
    }

    /**
     * Set default params when do not exist.
     *
     * @since 2.15.0
     */
    private function setComponentDefaultParams()
    {
        if ($this->componentDefaultSettings && $this->componentParams === null) {
            $component = $this->getInstalledExtension($this->componentName);
            if ($component && $component->params && $component->params != '{}') {
                // If there are component params set, make sure that all have a set value
                $this->componentParams = new Joomla\Registry\Registry($component->params);

                foreach ($this->componentDefaultSettings as $settingName => $settingValue) {
                    if ($this->componentParams->get($settingName, null) == null) {
                        // We set the 'sef_urls_for_cf' to 0 when updating from old versions, in order not break the existing url format.
                        if ($settingName == 'sef_urls_for_cf' && version_compare('2.15.0',
                                $this->getParam('version')) > 0) {
                            $this->componentParams->set($settingName, $settingValue);
                        }
                    }
                }
            }
        }
    }

    /**
     * Save the params to the db
     *
     * @since 2.15.0
     */
    private function saveComponentParams()
    {
        if ($this->componentParams) {
            // Put the params in the db
            try {
                /** @var DatabaseDriver $db */
                $db = Factory::getDbo();
                $query = $db->getQuery(true)
                            ->update('#__extensions')
                            ->set($db->quoteName('params') . ' = ' . $db->quote($this->componentParams->toString()))
                            ->where('element = ' . $db->quote($this->componentName));
                $db->setQuery($query);
                $db->execute();
            } catch (\Exception $e) {
                // Suck it. It works without params.
            }
        }
    }

    /**
     * Runs after installation
     *
     * @param string $type
     * @param $parent
     *
     * @since 2.0.0
     */
    public function postflight($type, $parent)
    {

        if ($type == 'install') {
            $this->enableExtensions();
        }
        //update
        elseif ($type == 'update') {
            if (!class_exists( 'cfHelper' )) {
                require(JPATH_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_customfilters'.DIRECTORY_SEPARATOR.'helpers'.DIRECTORY_SEPARATOR.'cfhelper.php');
            }
            require_once JPATH_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_customfilters' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'UpdateManager.php';
            \Breakdesigns\Customfilters\Admin\Model\UpdateManager::getInstance()->refreshUpdateSite();

            $this->saveComponentParams();
        }


        // There is also the 'uninstall'
        if ($type == 'install' || $type == 'update') {
            //copy the files to the joomla images folder
            $src_dir = JPATH_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_customfilters' . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'images' . DIRECTORY_SEPARATOR . 'category_tree';
            $dst_dir = JPATH_SITE . DIRECTORY_SEPARATOR . 'images' . DIRECTORY_SEPARATOR . 'stories' . DIRECTORY_SEPARATOR . 'customfilters';
            $this->recurse_copy($src_dir, $dst_dir);

            $this->showMessage($this->getPackageExtensions($parent->getParent()->manifest), $type);
        }
    }

    /**
     * Enable modules and plugins after installing them
     */
    private function enableExtensions()
    {
        foreach ($this->extensionsToEnable as $ext)
        {
            $this->enableExtension($ext[0], $ext[1], $ext[2], $ext[3]);
        }
    }

    /**
     * Enable an extension
     *
     * @param   string   $type    The extension type.
     * @param   string   $name    The name of the extension (the element field).
     * @param   integer  $client  The application id (0: Joomla CMS site; 1: Joomla CMS administrator).
     * @param   string   $group   The extension group (for plugins).
     */
    private function enableExtension($type, $name, $client = 1, $group = null)
    {
        try
        {
            $db    = Factory::getDbo();
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set($db->quoteName('enabled') . ' = ' . $db->quote(1))
                ->where('type = ' . $db->quote($type))
                ->where('element = ' . $db->quote($name));
        }
        catch (\Exception $e)
        {
            return;
        }


        switch ($type)
        {
            case 'plugin':
                // Plugins have a folder but not a client
                $query->where('folder = ' . $db->quote($group));
                break;

            case 'language':
            case 'module':
            case 'template':
                // Languages, modules and templates have a client but not a folder
                $client = ApplicationHelper::getClientInfo($client, true);
                $query->where('client_id = ' . (int) $client->id);
                break;

            default:
            case 'library':
            case 'package':
            case 'component':
                // Components, packages and libraries don't have a folder or client.
                // Included for completeness.
                break;
        }

        try
        {
            $db->setQuery($query);
            $db->execute();
            //$this->extensionsStatus->$type[] = array('name' => $name, 'group' => $group, 'result' => true);
        }
        catch (\Exception $e)
        {
        }
    }

    /**
     * copy all $src to $dst folder and remove it
     *
     * @param String $src path
     * @param String $dst path
     */
    private function recurse_copy($src, $dst)
    {
        $dst_exist = Folder::exists($dst);
        jimport('joomla.filesystem.folder');
        if (!$dst_exist) $dst_exist = Folder::create($dst);
        $dir = opendir($src);

        if (is_resource($dir) && $dst_exist) {
            jimport('joomla.filesystem.file');
            while (false !== ($file = readdir($dir))) {
                if (($file != '.') && ($file != '..')) {
                    if (is_dir($src . DIRECTORY_SEPARATOR . $file)) {
                        $this->recurse_copy($src . DIRECTORY_SEPARATOR . $file, $dst . DIRECTORY_SEPARATOR . $file);
                    } else {
                        if (File::exists($dst . DIRECTORY_SEPARATOR . $file)) {
                            //do nothing
                        }
                        if (!File::move($src . DIRECTORY_SEPARATOR . $file, $dst . DIRECTORY_SEPARATOR . $file)) {
                            //do nothing
                        }
                    }
                }
            }
        }
        if (is_resource($dir)) closedir($dir);
        if (is_dir($src)) Folder::delete($src);
    }

    /**
     * Displays post installation messages
     *
     * @param $type
     */
    private function showMessage($includedFiles, $type)
    {
        $language = Factory::getLanguage();
        $language->load('com_customfilters');
        $rows = 0;
        ?>
        <?php
        //After installation steps
        if ($type == 'install'):?>
            <div class="clr clearfix"></div>
            <div class="alert alert-info">
                Please read the <a style="text-decoration: underline"
                        href="https://breakdesigns.net/extensions/joomla/custom-filters/documentation-pro/49-basic-steps-after-the-installation"
                        target="_blank">Basic Steps After the Installation</a>, before proceeding
            </div>
        <?php
        endif;
        ?>
        <?php
        //if update messages
        if (!empty($this->printed_messages)) {
        ?>
            <div class="clr clearfix"></div>

        <h3><?php echo Text::_('COM_CUSTOMFILTERS_UPDATE_MESSAGES'); ?></h3>
        <div id="system-message-container">

            <?php
            foreach ($this->printed_messages as $message) {?>
                <div class="alert alert-info"><?php echo $message ?></div>
            <?php } ?>
        </div>
    <?php } ?>

        <div class="clr clearfix"></div>
        <h3><?php echo Text::_('Included Extensions'); ?></h3>
        <table class="adminlist table table-striped" width="100%">
            <thead>
            <tr>
                <th class="title" width="30%"><?php echo Text::_('COM_CUSTOMFILTERS_EXTENSION'); ?></th>
                <th class="title" width="30%"><?php echo Text::_('Type'); ?></th>
                <th><?php echo Text::_('JGLOBAL_DESCRIPTION'); ?></th>
            </tr>
            </thead>
            <tfoot>
            <tr>
                <td colspan="3"></td>
            </tr>
            </tfoot>
            <tbody>

            <?php if (count($includedFiles)): ?>
                <?php foreach ($includedFiles as $file): ?>
                    <tr class="row<?php echo(++$rows % 2); ?>">
                        <td><?php echo $file['name']; ?></td>
                        <td><?php echo $file['type']; ?></td>
                        <td><?php echo $file['description'];?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
        <?php
    }
}
?>
