<?php
/***
 * @package   customfilters
 * @copyright Copyright (C) 2012-2023 breakdesigns.net . All rights reserved.
 * @license   GNU General Public License version 2 or later; see LICENSE.txt
 * @author    Sakis Terz
 */

use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die();

/**
 * Class that fetches an array of items or an array of item properties by their ids.
 * The class is being extended by each type of filter (e.g. Category, Manufacturer) to make the database queries calls
 */
class CfItemsHelper
{
    /**
     * @var array
     * @since 2.15.0
     */
    protected static $instance = [];

    /**
     * @var array
     * @since 2.14.0
     */
    protected $items = [];

    /**
     * @return CfItemsHelper|mixed
     * @since 2.15.0
     */
    public static function getInstance()
    {
        $type = static::class;
        if (!isset(self::$instance[$type])) {
            self::$instance[$type] = new $type();
        }

        return self::$instance[$type];
    }

    /**
     * @param   array  $ids
     *
     * @return bool|string
     * @since 2.15.0
     */
    public function get(string $attribute, array $ids, $dataType = 'ids') : array
    {
        $names = [];

		// Filter empty vars
	    $ids = array_filter($ids);

		if (!$ids) {
			return $names;
		}

        $items = $this->getItems($ids, $dataType);
        foreach ($items as $id => $item) {
            if (isset($item->$attribute)) {
                $names[$id] = $item->$attribute;
            }
        }

        return $names;
    }

    /**
     * B/C function.
     * Seems that VMUkit is overriding our view and calls old functions.
     *
     * @param ...$params
     *
     * @return array
     * @since 2.14.0
     */
	public static function getNames(...$params)
	{
        $ids = reset($params);
		$instance = self::getInstance();
		return $instance->get('name', $ids);
	}

	/**
	 * Get the items by id
	 *
	 * @param array  $ids
	 * @param string $valuesType
	 *
	 * @return array
	 * @since 2.15.0
	 */
    public function getItems(array $ids, string $valuesType= 'ids')
    {
        if (empty($ids)) {
            return [];
        }

		$idsHex = false;

	    // Create a hash for strings
	    if (!preg_match('/^[-+]?\d*\.?\d+$/', reset($ids))) {
		    $ids = array_map('bin2hex', $ids);
		    $idsHex = true;
	    }

        // Search in the cache if already fetched
        $cachedItems = [];
        if (!empty($this->items)) {
            foreach ($ids as $id) {
                if (isset($this->items[$id])) {
                    $cachedItems[$id] = $this->items[$id];
                }
            }

            // Asks all the cached items
            if (count($ids) === count($cachedItems)) {
                return $cachedItems;
            }
        }

        $remainingIds = array_diff($ids, array_keys($cachedItems));

        // Make sure that the array keys returned are hashed (bin2hex), if strings
        $itemsFromStorage = $this->getItemsFromStorage($remainingIds, $valuesType, $idsHex);
        $this->items = $this->items + $itemsFromStorage;

        // Sort them by ids
        $toBeReturned = [];
        foreach ($ids as $id) {
            if (isset($this->items[$id])) {
                $toBeReturned[$id] = $this->items[$id];
            }
        }

        return $toBeReturned;
    }
}