/**
 * @package CustomfieldsforallBasebase
 * @copyright Copyright (C)2023 breakdesigns.net . All rights reserved.
 * @license GNU General Public License version 2 or later; see LICENSE.txt
 */

'use strict';

// top namespace
window.CF4All = window.CF4All || {};

CF4All.Updater = {
    async fetchResource(url, params) {
        return fetch(url, params).then(response => {
            if (!response.ok) {
                throw new Error(`CF4All error! status: ${response.status} - ${response.statusText}`);
            } else {
                return response.text();
            }
        })
            .catch(e => {
                console.error(`CF4All error calling: "${url}": ` + e.message);
            });
    },
    appendResponse(responseText, target) {
        let targetElement = document.querySelector(target);

        // Create a dummy object and inject the response in that, so that we can handle the response as dom.
        let tmpResponseWrapper = document.createElement("cf4all-temp-results-wrapper");
        tmpResponseWrapper.innerHTML = responseText;
        let responseElement = tmpResponseWrapper.querySelector(target);

        // If no target element the response is empty
        if (responseElement) {
            // We need to destroy existing select2 elements. Will be re-created
            if (window.jQuery && window.jQuery.fn.select2) {
                jQuery(".cf_parent_select.select2-hidden-accessible").select2("destroy");
            }
            targetElement.insertAdjacentHTML('beforeend', responseElement.innerHTML);
        } else {
            targetElement.innerHTML = '';
        }

        // Apply js
        let js = [].slice.call(tmpResponseWrapper.querySelectorAll('script'));
        js.forEach((jsElement) => {
            let script = document.createElement('script');
            script.innerHTML = jsElement.innerHTML;
            targetElement.append(script);
        });

        tmpResponseWrapper.remove();
        const responseTrimmed = responseElement.innerHTML.trim();
        return Promise.resolve(responseTrimmed.length);
    },
};

CF4All.Customvalues = {
    page : 1,
    updateURL: 'index.php?option=com_ajax&tmpl=component&view=custom',
    loadedEntirely : false,

    loadCollection : function () {
        const customFieldOptions = Joomla.getOptions('cf4all.custom');
        if (!this.loadedEntirely) {
            // Show loader
            const loader = document.querySelector('.cf_loadmore_loader');
            loader.classList.remove('hide', 'd-none');

            this.page++;
            let url = this.updateURL + '&plugin=getCustomValues&format=raw&virtuemart_custom_id=' + customFieldOptions.virtuemart_custom_id + '&page=' +  this.page + '&parentPlugin=' + customFieldOptions.parentCustomField;
            CF4All.Updater.fetchResource(url).then((response) => {
                if (response) {
                    CF4All.Updater.appendResponse(response, '.cf_values_wrapper').then((responseLength) => {
                        // Hide the loader
                        loader.classList.add('hide', 'd-none');
                        if(responseLength == 0) {
                            this.disableFurtherLoading();
                        }
                        this.addEventListenerOnElements();
                    });
                }
            })
        }
    },
    save : function(bodyString, elementToUpdate) {
        if(!bodyString) {
            return false;
        }
        console.log('Start Saving');
        const customFieldOptions = Joomla.getOptions('cf4all.custom');
        const fetchParams = {
            method: 'POST',
            cache: 'no-cache',
            headers: {
                //'Content-Type': 'application/json'
                'Content-Type': 'application/x-www-form-urlencoded',
                'charset': 'UTF-8'
            },
        }
        fetchParams.body = bodyString;
        let url = this.updateURL + '&plugin=saveCustomValue&format=json&virtuemart_custom_id=' + customFieldOptions.virtuemart_custom_id + '&parentPlugin=' + customFieldOptions.parentCustomField;
        CF4All.Updater.fetchResource(url, fetchParams).then((response) => {
            const responseJson = JSON.parse(response);
            if(responseJson.data && Array.isArray(responseJson.data)) {
                responseJson.data = JSON.parse(responseJson.data.shift());
                const inputs = elementToUpdate.querySelectorAll('input');

                inputs.forEach((innerInput) => {
                    // If it has value go to the next
                    if(innerInput.value && innerInput.value!=0) {
                        return;
                    }
                    const inputName = innerInput.name;
                    let responseData = responseJson.data;
                    let responseArray = [];
                    if(inputName) {
                        const inputParts = inputName.split('[');
                        for(let i=2; i<inputParts.length; i++) {
                            const inputPart = inputParts[i].replace(new RegExp(/\]$/), "");

                            // Do not update these inputs
                            if (!inputPart || inputPart == 'customsforall_value_name' || inputPart == 'customsforall_value_label') {
                                continue;
                            }

                            if(responseArray[inputPart]) {
                                responseData = responseArray;
                            }

                            if(typeof responseData[inputPart] === 'object' || Array.isArray(responseData[inputPart])) {
                                responseArray = responseData[inputPart];
                                continue;
                            }
                            if(responseData[inputPart]) {
                                innerInput.value = responseData[inputPart];
                            }
                        }
                    }
                });
                this.showSuccessIcon(elementToUpdate);
            } else {
                this.showFailureIcon(elementToUpdate);
            }
        }).catch((e) => {
            console.error(e);
            this.showFailureIcon(elementToUpdate);
        }) ;
    },

    delete: function (value_id) {
        value_id = parseInt(value_id);
        if(!value_id) {
            return false;
        }

        const customFieldOptions = Joomla.getOptions('cf4all.custom');
        let url = this.updateURL + '&plugin=deleteCustomValue&format=json&virtuemart_custom_id=' + customFieldOptions.virtuemart_custom_id + '&parentPlugin=' + customFieldOptions.parentCustomField+'&value_id=' + value_id;
        CF4All.Updater.fetchResource(url).then((response) => {});
    },

    order: function (value_id, prevPosition, newPosition) {
        value_id = parseInt(value_id);
        newPosition = parseInt(newPosition);
        if(!value_id) {
            return false;
        }

        const valueInput = document.querySelector('input[name$="[customsforall_value_id]"][value="'+value_id+'"]');
        if(!valueInput) {
            return false;
        }
        const elementToUpdate = valueInput.closest('.cf4all-value-wrapper');
        const customFieldOptions = Joomla.getOptions('cf4all.custom');
        let url = this.updateURL + '&plugin=orderCustomValue&format=json&virtuemart_custom_id=' + customFieldOptions.virtuemart_custom_id + '&parentPlugin=' +
            customFieldOptions.parentCustomField+'&value_id=' + value_id+'&newposition=' +newPosition+'&prevposition='+parseInt(prevPosition);
        CF4All.Updater.fetchResource(url).then((response) => {
            const responseJson = JSON.parse(response);
            if(responseJson.data && Array.isArray(responseJson.data)) {
                responseJson.data = JSON.parse(responseJson.data.shift());
                if(responseJson.data.result = 1) {
                    this.showSuccessIcon(elementToUpdate);
                    return true;
                }
            }
            this.showFailureIcon(elementToUpdate);
        });
    },

    showSuccessIcon: function (parentElement) {
        // Show success icon. The reason we do not use classes here (e.g. hide, d-none) is because they are overwritten by the VM template's css in J3
        const successIcon = parentElement.querySelector('.icon-checkmark');
        if(successIcon) {
            successIcon.style.display = 'inline-block';

            // And after a while hide it, so that can be shown again
            setTimeout(() => {
                successIcon.style.display = 'none';
            }, 3000)
        }
    },
    showFailureIcon: function (parentElement) {
        // Show failure icon. The reason we do not use classes here (e.g. hide, d-none) is because they are overwritten by the VM template's css in J3
        const failureIcon = parentElement.querySelector('.icon-cancel');
        if(failureIcon) {
            failureIcon.style.display = 'inline-block';

            // And after a while hide it, so that can be shown again
            setTimeout(() => {
                failureIcon.style.display = 'none';
            }, 3000)
        }
    },

    disableFurtherLoading: function () {
        this.loadedEntirely = true;
        document.querySelector('.cf_loadmore_btn').remove();
    },

    addEventListenerOnElements: function (CfWrapper) {
        CfWrapper = CfWrapper && typeof CfWrapper === 'object' ? CfWrapper : document.querySelector('.cf_values_wrapper');

        let selects = [].slice.call(CfWrapper.querySelectorAll('select'));
        selects.forEach((select) => {
            select.addEventListener('change', (event) => {
                const parentWrapper = select.closest('.cf4all-value-wrapper');

                const valueInput = parentWrapper.querySelector('input[name$="[customsforall_value_id]"]');
                // Do not save when no value is saved (parent cf selection)
                if(valueInput && parseInt(valueInput.value) > 0) {
                    let postBodyString = CF4All.Customvalues.getBodyString(parentWrapper);

                    CF4All.Customvalues.save(postBodyString, parentWrapper);
                }
            });
        })

        let valueInputs = [].slice.call(CfWrapper.querySelectorAll('input:not([type=hidden])'));
        valueInputs.forEach((input) => {
            // Input events
            input.addEventListener('blur', (event) => {
                if (!input.value) {
                    console.log('Aborted');
                    return;
                }
                const parentWrapper = input.closest('.cf4all-value-wrapper');
                let postBodyString = CF4All.Customvalues.getBodyString(parentWrapper);

                CF4All.Customvalues.save(postBodyString, parentWrapper);
            });
        });

        const deleteButtons = [].slice.call(CfWrapper.querySelectorAll('.customsforall_delete_btn'));
        deleteButtons.forEach((deleteButton) => {
            deleteButton.addEventListener('click', (event) => {
                const button = event.target;
                const parentWrapper = deleteButton.closest('.cf4all-value-wrapper');
                const valueInput = parentWrapper.querySelector('input[name$="[customsforall_value_id]"]');

                if(valueInput) {
                    CF4All.Customvalues.delete(valueInput.value);
                }

            })
        });

        // Change on chosen select
        if(typeof jQuery != 'undefined') {
            jQuery('.cf_parent_select').chosen().change(function() {
                const parentWrapper = this.closest('.cf4all-value-wrapper');

                const valueInput = parentWrapper.querySelector('input[name$="[customsforall_value_id]"]');
                // Do not save when no value is saved (parent cf selection)
                if(valueInput && parseInt(valueInput.value) > 0) {
                    let postBodyString = CF4All.Customvalues.getBodyString(parentWrapper);
                    CF4All.Customvalues.save(postBodyString, parentWrapper);
                }
            });
        }
    },

    getBodyString: function (parentWrapper) {
        const inputs = [].slice.call(parentWrapper.querySelectorAll('input'));
        const selects = [].slice.call(parentWrapper.querySelectorAll('select'));

        const elementsWitValue = inputs.concat(selects);
        let postBodyString = '';
        /*
         * We need to get all inputs belonging to that value
         * E.g. Can be multi-lingual or multi-color
         */
        elementsWitValue.forEach((innerElement) => {
            if (innerElement.name) {
                postBodyString += innerElement.name + '=' + innerElement.value + '&';
            }
        });
        return postBodyString;
    }
};


document.addEventListener('DOMContentLoaded', () => {
    const loadMoreBtn = document.querySelector('.cf_loadmore_btn');
    if(loadMoreBtn) {
        loadMoreBtn.addEventListener('click', (event) => {
            CF4All.Customvalues.loadCollection();
        });
    }

    CF4All.Customvalues.addEventListenerOnElements();

    // Called after dragging an element to change it's position
    if(typeof jQuery != 'undefined' && typeof jQuery.ui != 'undefined') {
        jQuery('.sortable').on("sortstart", ( event, ui ) => {
            // creates a temporary attribute on the element with the old index
            ui.item.attr('data-previndex', ui.item.index());
        });
        jQuery('.sortable').on("sortupdate", ( event, ui ) => {
            const newPosition = parseInt(ui.item.index()) + 1;
            const prevPosition = parseInt(ui.item.attr('data-previndex')) + 1;
            const valueId = ui.item.find('input[name$="[customsforall_value_id]"]').val();
            CF4All.Customvalues.order(valueId, prevPosition, newPosition);
        });
    }

    // Triggered when a new color input is added (multi-color value)
    document.addEventListener('addedColorInput', (event) => {
        const wrapper = event.detail.wrapper;
        if(wrapper) {
            CF4All.Customvalues.addEventListenerOnElements(wrapper);
            // Save the default value on new color input
            const bodyString = CF4All.Customvalues.getBodyString(wrapper);
            CF4All.Customvalues.save(bodyString, wrapper);
        }
    });

    // Triggered when a color input is removed (multi-color value)
    document.addEventListener('removedColorInput', (event) => {
        const wrapper = event.detail.wrapper;
        if(wrapper) {
            // Save the removed color inputs
            const bodyString = CF4All.Customvalues.getBodyString(wrapper);
            CF4All.Customvalues.save(bodyString, wrapper);
        }
    });
});