<?php
/**
 * @package      customfilters
 * @copyright    Copyright (C) 2012-2022 breakdesigns.net . All rights reserved.
 * @license      GNU General Public License version 2 or later; see LICENSE.txt
 * @subpackage   mod_cf_filtering
 */

// no direct access
defined('_JEXEC') or die;


class CfFilter
{
    const DISPLAY_SELECT = '1';

    const DISPLAY_RADIO = '2';

    const DISPLAY_CHECKBOX = '3';

    const DISPLAY_LINK = '4';

    const DISPLAY_INPUT_TEXT = '5';

    const DISPLAY_RANGE_SLIDER = '6';

    const DISPLAY_IMAGE_LINK = '7';

    const DISPLAY_RANGE_DATES = '8';

    const DISPLAY_COLOR_BUTTON = '9';

    const DISPLAY_COLOR_BUTTON_MULTI = '10';

    const DISPLAY_BUTTON = '11';

    const DISPLAY_BUTTON_MULTI = '12';

    /**
     * The displays in array
     * Those keys are used as a reference to the sublayout file names
     *
     * @var array
     * @since 2.8.0
     */
    public $displays = [
        'select' => self::DISPLAY_SELECT,
        'radio' => self::DISPLAY_RADIO,
        'checkbox' => self::DISPLAY_CHECKBOX,
        'link' => self::DISPLAY_LINK,
        'input_text' => self::DISPLAY_INPUT_TEXT,
        'range_slider' => self::DISPLAY_RANGE_SLIDER,
        'image_link' => self::DISPLAY_IMAGE_LINK,
        'range_dates' => self::DISPLAY_RANGE_DATES,
        'color_button' => self::DISPLAY_COLOR_BUTTON,
        'color_button_multi' => self::DISPLAY_COLOR_BUTTON_MULTI,
        'button' => self::DISPLAY_BUTTON,
        'button_multi' => self::DISPLAY_BUTTON_MULTI
    ];

    /**
     * @var string
     * @since 2.8.0
     */
    protected $var_name;

    /**
     * @var array
     * @since 2.8.0
     */
    protected $display = [self::DISPLAY_SELECT];

    /**
     * @var string
     * @since 2.8.0
     */
    protected $header = '';

    /**
     * @var string
     * @since 2.9.2
     */
    protected $description = '';

    /**
     * @var string
     * @since 2.8.0
     */
    protected $type = 'string';

    /**
     * @var string
     * @since 2.8.0
     */
    protected $clearType = 'all';

    /**
     * @var bool
     * @since 2.8.0
     */
    protected $expanded = true;

    /**
     * @var bool
     * @since 2.8.0
     */
    protected $smartSearch = false;

    /**
     * @var int
     * @since 2.13.0
     */
    protected $showMoreAfter = 0;

    /**
     * @var bool
     * @since 2.13.0
     */
    protected $showMoreState;

    /**
     * @var bool
     * @since 2.8.0
     */
    protected $counter = true;

    /**
     * @var array
     * @since 2.8.0
     */
    protected $options = [];

    /**
     * @var array
     * @since 2.8.0
     */
    protected $activeTree;

    /**
     * @var int
     * @since 2.13.0
     */
    protected $optionsSize;

    /**
     * Store the values which are selected but disabled
     *
     * @var array
     * @since 2.8.0
     */
    protected $inactive_select_opt;

    /**
     * Get the name of the filter
     *
     * @return string
     * @since 2.8.0
     */
    public function getVarName()
    {
        if (empty($this->var_name)) {
            throw new \RuntimeException('No name is defined for the filter');
        }

        return $this->var_name;
    }

    /**
     * @param $var_name
     *
     * @return CfFilter
     * @since 2.8.0
     */
    public function setVarName($var_name)
    {
        $this->var_name = (string)$var_name;

        return $this;
    }

    /**
     * Get the name of the filter
     *
     * @return string
     * @since 2.8.0
     */
    public function getDisplay()
    {
        return $this->display;
    }

    /**
     * Set the display type of the filter
     *
     * @param   int  $display
     *
     * @return $this
     * @since 2.8.0
     */
    public function setDisplay($display)
    {
        if (!is_array($display)) {
            $display = explode(',', $display);
        }
        $display = array_map('intval', $display);
        $this->display = $display;

        return $this;
    }

    /**
     * @return string
     * @since 2.8.0
     */
    public function getHeader()
    {
        return $this->header;
    }

    /**
     * @param $header
     *
     * @return $this
     * @since 2.8.0
     */
    public function setHeader($header)
    {
        $this->header = $header;

        return $this;
    }

    /**
     * Get a description for the filter
     *
     * @return string
     * @since 2.9.2
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Set a description for the filter
     *
     * @param $description
     *
     * @return $this
     * @since 2.9.2
     */
    public function setDescription($description)
    {
        $this->description = $description;

        return $this;
    }

    /**
     * @return string
     * @since 2.8.0
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param $type
     *
     * @return $this
     * @since 2.8.0
     */
    public function setType($type)
    {
        $this->type = (string)$type;

        return $this;
    }

    /**
     * @return string
     * @since 2.8.0
     */
    public function getClearType()
    {
        return $this->clearType;
    }

    /**
     * @param $clearType
     *
     * @return $this
     * @since 2.8.0
     */
    public function setClearType($clearType)
    {
        $this->clearType = (string)$clearType;

        return $this;
    }

    /**
     * @return bool
     * @since 2.8.0
     */
    public function getExpanded()
    {
        return $this->expanded;
    }

    /**
     * @param   bool  $expanded
     *
     * @return $this
     * @since 2.8.0
     */
    public function setExpanded($expanded)
    {
        $this->expanded = (bool)$expanded;

        return $this;
    }

    /**
     * @return bool
     * @since 2.13.0
     */
    public function getShowMoreState(): bool
    {
        if ($this->showMoreState === null) {
            $this->showMoreState = false;
            if ($this->getOptionsSize() > $this->getShowMoreAfter()) {
                /*
                 * If there is a selection after the 'show more' threshold the list should be visible.
                 */
                $options = array_values($this->getOptions());
                $counter = 1;
                foreach ($options as $option) {
                    if ($counter > $this->getShowMoreAfter() && $option->selected) {
                        $this->showMoreState = true;
                        break;
                    }
                    if ($option->type != 'clear' && (!isset($option->level) || (isset($option->level) && $option->level == 0))) {
                        $counter++;
                    }
                }
            }
        }

        return $this->showMoreState;
    }

    /**
     * @param   bool  $showMoreState
     *
     * @return $this
     * @since 2.13.0
     */
    public function setShowMoreState(bool $showMoreState)
    {
        $this->showMoreState = $showMoreState;

        return $this;
    }

    /**
     * Get the amount of the real options of that filter
     *
     * @param   bool  $onlyTopLevel  Count only the top level options, in case of nesting
     *
     * @return int
     * @since 1.3.0
     */
    public function getOptionsSize($onlyTopLevel = true)
    {
        if ($this->optionsSize === null) {
            $options = $this->getOptions();
            $this->optionsSize = count($options);
            $firstOption = reset($options);

            // We only want to count real options (not clears)
            if ($firstOption->type == 'clear') {
                $this->optionsSize--;
            }

            // Count only the top level options
            if ($onlyTopLevel) {
                foreach ($options as $option) {
                    if (isset($option->level) && $option->level > 0) {
                        $this->optionsSize--;
                    }
                }
            }
        }

        return $this->optionsSize;
    }

    /**
     * Get the options of the filter.
     *
     * @return array
     * @since 2.8.0
     */
    public function getOptions()
    {
        return $this->options;
    }

    /**
     * Set the options in an associative array by their ids.
     * In many cases we access the array by the option's id.
     *
     * @param   array  $options
     *
     * @return $this
     * @since 2.8.0
     */
    public function setOptions(array $options)
    {
        // reset it in case it had previous options set
        $this->options = [];
        /*
         * convert to object
         * @todo create a class for Option and define them properly in the 1st place
         */
        foreach ($options as $key => $option) {
            $option = (object)$option;
            $index = !empty($option->id) ? $option->id : $key;
            $this->options[$index] = $option;
        }

        return $this;
    }

    /**
     * @return int
     * @since 2.13.0
     */
    public function getShowMoreAfter(): int
    {
        return $this->showMoreAfter;
    }

    /**
     * @param   int  $showMoreAfter
     *
     * @return $this
     * @since 2.13.0
     */
    public function setShowMoreAfter($showMoreAfter)
    {
        $this->showMoreAfter = (int)$showMoreAfter;

        return $this;
    }

    /**
     * @return array
     * @since 2.8.0
     */
    public function getActiveTree()
    {
        return $this->activeTree;
    }

    /**
     * @param   array  $activeTree
     *
     * @return $this
     * @since 2.8.0
     */
    public function setActiveTree(array $activeTree)
    {
        $this->activeTree = $activeTree;

        return $this;
    }

    /**
     * @param   array  $inactive_select_opt
     *
     * @return $this
     * @since 2.8.0
     */
    public function setInactiveSelectedOptions(array $inactive_select_opt)
    {
        $this->inactive_select_opt = $inactive_select_opt;

        return $this;
    }

    /**
     * @return array
     * @since 2.8.0
     */
    public function getInactiveSelectedOptions()
    {
        return $this->inactive_select_opt;
    }

    /**
     * @return bool
     * @since 2.8.0
     */
    public function getSmartSearch()
    {
        return $this->smartSearch;
    }

    /**
     * @param   bool  $smartSearch
     *
     * @return $this
     * @since 2.8.0
     */
    public function setSmartSearch($smartSearch)
    {
        $this->smartSearch = (bool)$smartSearch;

        return $this;
    }

    /**
     * @return bool
     * @since 2.8.0
     */
    public function getCounter()
    {
        return $this->counter;
    }

    /**
     * @param $counter
     *
     * @return $this
     * @since 2.8.0
     */
    public function setCounter($counter)
    {
        $this->counter = (bool)$counter;

        return $this;
    }
}
