<?php

/**
 * @package		CustomfieldsforallBase
 * @copyright	Copyright (C)2014-2023 breakdesigns.net . All rights reserved.
 * @license		GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die('Restricted access');

use Breakdesigns\Plugin\System\Customfieldsforallbase\Model\Customfield;

/**
 * Block class that gives values to the layouts
 * @author sakis
 *
 */
class ProductRow
{

    /**
     *
     * @var Customfield
     */
    protected $custom;

    /**
     *
     * @var string
     */
    protected $pluginName;

    /**
     *
     * @var int
     */
    protected $product_id;

    /**
     *
     * @var int
     */
    protected $virtuemart_custom_id = 0;

    /**
     * Assoc array containing the values of the block
     *
     * @var array
     */
    protected $customfield_values = [];

    /**
     * 
     * @param string $pluginName
     * @param int $product_id
     * @param int $virtuemart_custom_id
     */
    public function __construct($pluginName, $product_id, $virtuemart_custom_id)
    {
        $this->pluginName = $pluginName;
        $this->product_id = $product_id;
        $this->virtuemart_custom_id = $virtuemart_custom_id;
    }

    /**
     * 
     * @return Customfield
     */
    public function getCustom()
    {
        if ($this->custom === null) {
            $this->custom = Customfield::getInstance($this->virtuemart_custom_id, $this->pluginName);
        }
        return $this->custom;
    }
    
    /**
     * 
     * @return int
     */
    public function getVirtuemartCustom_id()
    {
        return $this->virtuemart_custom_id;
    }
    
    /**
     * 
     * @return array
     */
    public function getCustomParams()
    {
        return $this->getCustom()->getCustomfieldParams($this->virtuemart_custom_id);
    }

    /**
     * Get custom field values
     *
     * @param   false   $with_parent
     * @param   int     $limitStart
     * @param   int     $limit
     * @param   string  $searchTerm
     * @param   array   $ids
     *
     * @return array|mixed
     * @since 2.0.0
     */
    public function getCustomValues($with_parent = false, $limitStart = 0, $limit = 60, $searchTerm = '', $ids = [])
    {
        $key = md5($with_parent . $limitStart . $limit . $searchTerm . implode(',', $ids));
        if (!isset($this->customfield_values[$key])) {
            $valueObjectList = $this->getCustom()->getCustomValues('', $limitStart, $limit, $searchTerm, $ids);

            if ($with_parent && !empty($this->getCustom()->getParentId())) {

                // Fetch only the needed parent ids
                $parentIds = [];
                foreach ($valueObjectList as $valueObject) {
                    if($valueObject->parent_id > 0 && !isset($parentIds[$valueObject->parent_id])) {
                        $parentIds[$valueObject->parent_id] = $valueObject->parent_id;
                    }
                }

                //get the parent custom field values
                $parentBlock = new ProductRow($this->pluginName, $this->product_id, $this->getCustom()->getParentId());
                $parentValueObjectList =  $parentBlock->getCustomValues(true, 6000, '', '', $parentIds);

                foreach ($valueObjectList as $obj) {
                    if ($obj->parent_id) {
                        //Avoid recursion. Can happen with price variants which assigned multiple times
                        if (strpos($obj->customsforall_value_name,
                                $parentValueObjectList[$obj->parent_id]->customsforall_value_name) === false) {
                            $obj->customsforall_value_name = $parentValueObjectList[$obj->parent_id]->customsforall_value_name . ' > ' . $obj->customsforall_value_name;
                        }
                    }
                }
            }
            // Return them as assoc. array
            $this->customfield_values[$key] = [];
            foreach ($valueObjectList as $valueObject) {
                $this->customfield_values[$key][$valueObject->customsforall_value_id] = $valueObject;
            }
        }
        return $this->customfield_values[$key];
    }
    
    /**
     * Get the values of a product
     *
     * @param int $customfield_id
     * @return array
     */
    public function getProductValues($customfield_id = 0)
    {
        $product_values = [];
        $product_valueIds = $this->getCustom()->getProductCustomValues($this->product_id, 'cf.customsforall_value_id', $customfield_id);
        if(!empty($product_valueIds)) {
            $withParent = $this->getCustom()->getParentId() > 0;
            $product_values = $this->getCustomValues($withParent, 0, 6000, '', $product_valueIds);
        }
        $product_values_new = [];

        // Create assoc array based on the value id
        foreach ($product_values as $product_value) {
            $product_values_new[$product_value->customsforall_value_id] = $product_value;
        }

        return $product_values_new;
    }

    /**
     * 
     * @param string $value
     * @return boolean
     */
    public function isHexColor($value)
    {
        $ishex = false;
        if (strpos($value, '#') !== false) {
            $ishex = true;
        }
        return $ishex;
    }
}