<?php
/**
 * @package CustomfieldsforallBasebase
 * @copyright   Copyright (C)2014-2023 breakdesigns.net . All rights reserved.
 * @license GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Breakdesigns\Plugin\System\Customfieldsforallbase;

defined('_JEXEC') or die('Restricted access');

require_once __DIR__ . '/bootstrap.php';

use Breakdesigns\Plugin\System\Customfieldsforallbase\Model\Customfield;
use Breakdesigns\Plugin\System\Customfieldsforallbase\Model\Language\CustomFieldsForAllLanguageHandlerFactory;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;

/**
 * Class RenderFields
 * @package Breakdesigns\Plugin\System\Customfieldsforallbase
 */
class RenderFields
{
    /**
     * RenderFields constructor.
     */
    public function __construct()
    {
        //load the lang files of custom fields for all, on display the fields
        if(!Factory::getLanguage()->load('plg_vmcustom_customfieldsforall')) {
            Factory::getLanguage()->load('plg_vmcustom_dependentcustomfieldsforall');
        }
    }

    /**
     * Fetch the datatype field renedered
     *
     * @param string $fieldname
     * @param int $virtuemart_custom_id
     * @param string $value
     * @return string
     */
    public function fetchDatatype($virtuemart_custom_id, $fieldname = 'data_type', $value = 'string')
    {

        $options_array = array(
            'string' => Text::_('PLG_CUSTOMSFORALL_STRING'),
            'color_hex' => Text::_('PLG_CUSTOMSFORALL_COLOR_HEX'),
            'int' => Text::_('PLG_CUSTOMSFORALL_INT'),
            'float' => Text::_('PLG_CUSTOMSFORALL_FLOAT')
        );

        if ($virtuemart_custom_id) {
            $html = $options_array[$value];
            $html .= '<input type="hidden" name="' . $fieldname . '" value="' . $value . '"/>';
        } else {
            $options_new_array = array();
            foreach ($options_array as $key => $val) {
                $myOpt = new \stdClass();
                $myOpt->text = $val;
                $myOpt->value = $key;
                $options_new_array[] = $myOpt;
            }
            $html = HTMLHelper::_('select.genericlist', $options_new_array, $fieldname, 'class="inputbox required"', 'value', 'text', $value);
        }

        return $html;
    }

    /**
     * Fetch the display types field renedered
     *
     * @param   string  $fieldname
     * @param   int     $virtuemart_custom_id
     * @param   string  $default
     * @param   bool    $singleSelect
     *
     * @return string
     * @todo make the dependency script work with multi-select fields (checkboxes)
     */
    public function fetchDisplaytypes($fieldname, $virtuemart_custom_id, $default, $singleSelect = false)
    {
        $displaytypes = array(
            'button' => Text::_('PLG_CUSTOMSFORALL_BTN'),
            'button_multi' => Text::_('PLG_CUSTOMSFORALL_BTN_MULTI'),
            'color' => Text::_('PLG_CUSTOMSFORALL_COLOR_BTN'),
            'color_multi' => Text::_('PLG_CUSTOMSFORALL_COLOR_BTN_MULTI'),
            'checkbox' => Text::_('PLG_CUSTOMSFORALL_CHECKBOXES'),
            'radio' => Text::_('PLG_CUSTOMSFORALL_RADIO_BTN'),
            'select' => Text::_('PLG_CUSTOMSFORALL_SELECT_LIST'),
        );

        // Remove the multi-select displays
        if ($singleSelect) {
            unset($displaytypes['button_multi']);
            unset($displaytypes['color_multi']);
            unset($displaytypes['checkbox']);
        }

        //assoc array containing the valid display types for each datatype
        $datatypes = array(
            'string' => array('display_types' => array('button', 'button_multi', 'color', 'color_multi', 'checkbox', 'radio', 'select')),
            'color_hex' => array('display_types' => array('button', 'button_multi', 'color', 'color_multi', 'checkbox', 'radio', 'select')),
            'int' => array('display_types' => array('button', 'button_multi', 'checkbox', 'radio', 'select')),
            'float' => array('display_types' => array('button', 'button_multi', 'checkbox', 'radio', 'select')),
        );

        if (!empty($virtuemart_custom_id)) {
            $customfield = Customfield::getInstance($virtuemart_custom_id);
            $custom_params = $customfield->getCustomfieldParams($virtuemart_custom_id);
            $datatype = $custom_params['data_type'];
        }

        $options = array();

        foreach ($displaytypes as $key => $value) {
            $option = array(
                'value' => $key,
                'text' => $value,
            );

            if (isset($datatype) && !empty($datatypes[$datatype])) {
                if (!in_array($key, $datatypes[$datatype]['display_types'])) {
                    $option['attr'] = array('disabled' => 'true');
                    if ($default == $key) $default = 'button';
                }
            }
            $options[] = $option;
        }

        $properties = array(
            'id' => 'displaytypes', // HTML id for select field
            'list.attr' => array('class' => 'inputbox required',),
            'option.value' => 'value', // key name for value in data array
            'option.text' => 'text', // key name for text in data array
            'option.attr' => 'attr', // key name for attr in data array
            'list.select' => $default, // value of the SELECTED field
        );

        $html = $result = \JHtmlSelect::genericlist($options, $fieldname, $properties);
        return $html;
    }

    /**
     * Function to fetch the custom value inputs
     *
     * @param   string  $fieldname
     * @param   int     $virtuemart_custom_id
     * @param   string  $value
     * @param   int     $row
     * @param   array   $custom_params
     * @param   int     $limitStart
     * @param   int     $limit
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     */
    public function fetchCustomvalues($fieldname, $virtuemart_custom_id, $value = '', $row = 0, $custom_params = [], $limitStart = 0, $limit = 0)
    {
        $languageFactory = new CustomFieldsForAllLanguageHandlerFactory();
        $languageHandler = $languageFactory->get();
        $languages = $languageHandler->getLanguages();

        $app = Factory::getApplication();
        $jinput = $app->input;
        $view = $jinput->get('view', '', 'STRING');
        $is_js_loaded = $jinput->get('scripts_loaded', false, 'BOOLEAN');
        $existing_values = array();
        $parent_id = isset($custom_params['parent_id']) ? $custom_params['parent_id'] : 0;
        $customfield = Customfield::getInstance($virtuemart_custom_id);
        $customfield->loadStylesScripts();
        $data_type = !empty($custom_params['data_type']) ? $custom_params['data_type'] : 'string';

        $class = 'input';
        $html = '';

        if (empty($limit)) {
            // Set a default one in case the ajax plugin is disabled.
            $limit = 200;
            // Get the list size from the ajax plugin's setting
            $ajaxCF4All = PluginHelper::getPlugin('ajax', 'customfieldsforall');
            if ($ajaxCF4All) {
                $params = new Registry($ajaxCF4All->params);
                $limit = $params->get('limit', 60);
            }
        }

        if (!empty($virtuemart_custom_id) && $view == 'custom') {
            $existing_values = $customfield->getCustomValues('', $limitStart, $limit);
        }

        //if there are parents load the previous level as well
        if ($parent_id) {
            $customfieldParent = Customfield::getInstance($parent_id, $customfield->getPluginName());
            $parent_options = [];

            if($existing_values) {
                $parentIds = [];
                foreach ($existing_values as $existingValue) {
                    if($existingValue->parent_id > 0) {
                        $parentIds[] = $existingValue->parent_id;
                    }
                }

                $parent_options = $customfieldParent->getCustomValues('', 0, 8000, '', $parentIds);
                $parentCustomParams = $customfieldParent->getCustomfieldParams();

                // Check if there is 1 more level
                if (!empty($parent_options) && !empty($parentCustomParams['parent_id'])) {
                    $parentParentIds = [];

                    foreach ($parent_options as $parentOption) {
                        if ($parentOption->parent_id > 0) {
                            $parentParentIds[] = $parentOption->parent_id;
                        }
                    }
                    $customfieldParentParent = Customfield::getInstance($parentCustomParams['parent_id'], $customfield->getPluginName());
                    $parent_parent_options = $customfieldParentParent->getCustomValues('', 0, 8000, '', $parentParentIds);

                    foreach ($parent_options as &$option) {
                        if ($option->parent_id) {
                            if (strpos($option->customsforall_value_name, $parent_parent_options[$option->parent_id]->customsforall_value_name) === false) {
                                $option->customsforall_value_name = $parent_parent_options [$option->parent_id]->customsforall_value_name . ' > ' . $option->customsforall_value_name;
                            }
                        }
                    }
                }
            }
        }

        if ($data_type == 'color_hex') {
            $class .= ' color {required:false}';
        }

        // load the js scripts
        if (!empty($custom_params) && !$is_js_loaded && $limitStart == 0) {
            $scriptCommands = '';
            // if the color script does not exist we should load the script in the returned html
            if ($data_type == 'color_hex') {
                $script_path = CF4ALLBASE_PLUGIN_PATH . DIRECTORY_SEPARATOR . 'view' . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'js' . DIRECTORY_SEPARATOR . 'jscolor' . DIRECTORY_SEPARATOR . 'jscolor.min.js';
                $scriptCommands ='JSColor.init();';
            }

            // Load the script
            if(isset($script_path)) {
                ob_start();
                include($script_path);
                $script = ob_get_contents();
                $script .= $scriptCommands;
                $html .= '<script type="text/javascript">' . $script . '</script>';
                ob_end_clean();
            }

            if(isset($css_path)) {
                ob_start();
                include($css_path);
                $css = ob_get_contents();
                $html .= '<style>' . $css . '</style>';
                ob_end_clean();
            }
        }

        ob_start();
        $custom_values_template_path = CF4ALLBASE_PLUGIN_PATH . DIRECTORY_SEPARATOR . 'Field' . DIRECTORY_SEPARATOR . 'templates' . DIRECTORY_SEPARATOR . 'customvalues.php';
        include($custom_values_template_path);
        $html .= ob_get_contents();
        ob_end_clean();
        return $html;
    }

    /**
     * Renders a list of custom fields of type custom fields for all
     * @param string $fieldname
     * @param array $optionList
     * @param int $default
     * @return string
     *
     * @since  3.0.5
     */
    public function fetchParentCustom($fieldname, $optionList, $default)
    {
        $options = array();

        foreach ($optionList as $obj) {
            $option = array(
                'value' => $obj->virtuemart_custom_id,
                'text' => htmlspecialchars($obj->custom_title, ENT_QUOTES),
            );
            if (!empty($default) && $obj->virtuemart_custom_id == $default) {
                $defaultOption = $obj;
                break;
            }

            $options[] = $option;
        }

        if (!empty($defaultOption)) {
            $html = '<input type="hidden" name="' . $fieldname . '" value="' . $default . '"/>';
            $html .= '<span>' . $defaultOption->custom_title . '</span>';
        } else {
            array_unshift($options, array('value' => 0, 'text' => Text::_('PLG_CUSTOMSFORALL_SELECT_AN_OPTION_FE')));

            $properties = array(
                'id' => 'parent_id', // HTML id for select field
                'list.attr' => array('class' => 'inputbox required',),
                'option.value' => 'value', // key name for value in data array
                'option.text' => 'text', // key name for text in data array
                'option.attr' => 'attr', // key name for attr in data array
                'list.select' => $default, // value of the SELECTED field
            );

            $html = \JHtmlSelect::genericlist($options, $fieldname, $properties);
        }
        return $html;
    }

    /**
     *
     * @return string
     */
    public function fetchJEDMessage()
    {
        $html = '<div class="help-block" style="clear:both; margin-top:5em;">Are you happy with it? Please post a rating and a review at the
            <a target="_blank" href="http://extensions.joomla.org/extensions/extension/extension-specific/virtuemart-extensions/custom-fields-for-all">Joomla! Extensions Directory</a></div>';
        return $html;
    }
}
