<?php
/**
 * @package     CustomfieldsforallBasebase
 *
 * @Copyright   Copyright © 2010-2023 Breakdesigns.net. All rights reserved.
 * @license     GNU General Public License 2 or later, see COPYING.txt for license details.
 */

use Joomla\CMS\Factory;

/**
 * Class PlgsystemcustomfieldsforallbaseInstallerScript
 * @since 1.1.1
 */
class PlgsystemcustomfieldsforallbaseInstallerScript
{
    /**
     * Add here the alter sql statements for dynamic update of the cascade tables (e.g. language tables)
     * @var string[]
     * @since 2.0.0
     */
    protected $languageTableSQL = ['2.0.0' => 'ALTER TABLE %s CHANGE `customsforall_value_label` `customsforall_value_label` TEXT'];

    /**
     * This runs before Joomla! installs or updates the package.
     *
     * @param string $type
     * @param \Joomla\CMS\Installer\Adapter\PluginAdapter $adapter
     * @return bool
     * @since 1.1.1
     */
    public function preflight($type, $adapter)
    {
        if ($type == 'uninstall') {
            /*
             * Drop the language tables.
             * Otherwise the values table cannot be dropped, due to the fact that is referenced by their FKs
             */
            $this->dropCascadeTables();
        } elseif ($type == 'update') {
            $installedVersion = $this->getParam('version');
            $updateVersions = array_keys($this->languageTableSQL);
            foreach ($updateVersions as $version) {
                if (version_compare($installedVersion, $version) == -1) {
                    $this->updateCascadeTables($this->languageTableSQL[$version]);
                }
            }
        }
        return true;
    }

    /**
     * Drop tables, which use Foreign keys to our main table (e.g. virtuemart_custom_plg_customsforall_values)
     *
     * If we do not drop them, the main table cannot be dropped as well (Truncate and Drop Cannot check if the referential integrity is breaking, by performing their actions)
     *
     * @param string $likeTableName
     * @return bool
     * @since 1.1.1
     */
    protected function dropCascadeTables($likeTableName = 'virtuemart_custom_plg_customsforall_values')
    {
        if (empty($likeTableName)) {
            return false;
        }
        $db = Factory::getDbo();
        $likeTableName = $db->getPrefix() . $likeTableName;
        $query = 'SHOW TABLES LIKE ' . $db->quote($likeTableName . '%');
        $db->setQuery($query);
        $tables = $db->loadColumn();

        foreach ($tables as $dbTableName) {
            // we do not want to drop the main table
            if ($dbTableName != $likeTableName) {
                $query = "DROP TABLE " . $dbTableName;
                $db->setQuery($query);
                try {
                    $db->execute();
                } catch (\RuntimeException $exception) {
                    throw new \RuntimeException(sprintf('Cannot drop the table: %s , because of database error:',
                        $dbTableName, $exception->getMessage()));
                }
            }
        }
        return true;
    }

    /**
     * Update tables, which use Foreign keys to our main table (e.g. virtuemart_custom_plg_customsforall_values).
     * These tables are dynamically updated. Hence not possible to update them using an update sql.
     *
     * @param $sqlStatement
     * @param string $likeTableName
     * @return bool
     * @since 2.0.0
     */
    protected function updateCascadeTables($sqlStatement, $likeTableName = 'virtuemart_custom_plg_customsforall_values')
    {
        if (empty($likeTableName)) {
            return false;
        }
        $db = Factory::getDbo();
        $likeTableName = $db->getPrefix() . $likeTableName;
        $query = 'SHOW TABLES LIKE ' . $db->quote($likeTableName . '%');
        $db->setQuery($query);
        $tables = $db->loadColumn();
        foreach ($tables as $dbTableName) {
            // we do not want to update the main table
            if ($dbTableName != $likeTableName) {
                $sql = sprintf($sqlStatement, $dbTableName);
                $db->setQuery($sql);
                try {
                    $db->execute();
                } catch (\RuntimeException $exception) {
                    throw new \RuntimeException(sprintf('Cannot update the table: %s , because of database error:',
                        $dbTableName, $exception->getMessage()));
                }
            }
        }
        return true;
    }

    /**
     * Get a variable from the manifest file (actually, from the manifest cache).
     *
     * @param $name
     * @return mixed
     * @since 2.0.0
     */
    private function getParam($name)
    {
        $db = Factory::getDbo();
        $db->setQuery('SELECT manifest_cache FROM #__extensions WHERE element = "customfieldsforallbase"');
        $manifest = json_decode($db->loadResult(), true);
        return $manifest[$name];
    }
}
