<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

class VPFrameworkUrl
{
	protected static $instances = array();
	
	public static function getInstance($uri = 'SERVER')
	{
		if (!isset(self::$instances[$uri]))
		{
			$uriObject = clone JUri::getInstance($uri);
			
			if ($uri != 'SERVER')
			{
				$uri   = str_replace('\\/', '/', $uri);
				$parts = self::parseUrl($uri);

				$uriObject->setScheme(!empty($parts['scheme']) ? $parts['scheme'] : null);
				$uriObject->setUser(!empty($parts['user']) ? $parts['user'] : null);
				$uriObject->setPass(!empty($parts['pass']) ? $parts['pass'] : null);
				$uriObject->setHost(!empty($parts['host']) ? $parts['host'] : null);
				$uriObject->setPort(!empty($parts['port']) ? $parts['port'] : null);
				$uriObject->setPath(!empty($parts['path']) ? $parts['path'] : null);
				$uriObject->setQuery(!empty($parts['query']) ? $parts['query'] : null);
				$uriObject->setFragment(!empty($parts['fragment']) ? $parts['fragment'] : null);
			}
			
			self::$instances[$uri] = $uriObject;
		}

		return self::$instances[$uri];
	}
	
	public static function parseURL($url)
	{
		preg_match('#^(?:([a-z][a-z0-9+.-]*+):(?=//))?(?://(?:(?:([^:@/]*+)(?::([^@/]*+))?@)?([^:/]*+)?(?::([^/]*+))?)?(?=/))?'
		        . '((?:/|^)[^?\#\n]*+)(?:\?([^\#\n]*+))?(?:\#(.*+))?$#i', $url, $matches);

		$parts = array();

		$parts['scheme']   = !empty($matches[1]) ? $matches[1] : null;
		$parts['user']     = !empty($matches[2]) ? $matches[2] : null;
		$parts['pass']     = !empty($matches[3]) ? $matches[3] : null;
		$parts['host']     = !empty($matches[4]) ? $matches[4] : null;
		$parts['port']     = !empty($matches[5]) ? $matches[5] : null;
		$parts['path']     = !empty($matches[6]) ? $matches[6] : '';
		$parts['query']    = !empty($matches[7]) ? $matches[7] : null;
		$parts['fragment'] = !empty($matches[8]) ? $matches[8] : null;

		return $parts;
	}
	
	public static function isInternal($url)
	{
		$urlObject = clone self::getInstance($url);

		if (preg_match('#^//#', $url))
		{
			$uriObject = self::getInstance();
			$urlObject->setScheme($uriObject->getScheme());
		}

		$urlBase = $urlObject->toString(array('scheme', 'user', 'pass', 'host', 'port', 'path'));
		$urlHost = $urlObject->toString(array('scheme', 'user', 'pass', 'host', 'port'));

		$base = JUri::base();

		if (!empty($base) && stripos($urlBase, $base) !== 0 && !empty($urlHost))
		{
			return false;
		}

		return true;
	}

	public static function isAbsolute($url)
	{
		return preg_match('#^http#i', $url);
	}

	public static function isRootRelative($url)
	{
		return preg_match('#^/[^/]#', $url);
	}

	public static function isProtocolRelative($url)
	{
		return preg_match('#^//#', $url);
	}

	public static function isPathRelative($url)
	{
		return self::isHttpScheme($url) 
		  && !self::isAbsolute($url)
		  && !self::isProtocolRelative($url)
		  && !self::isRootRelative($url);
	}

	public static function isSSL($url)
	{
		return preg_match('#^https#i', $url);
	}

	public static function isDataUri($url)
	{
		return preg_match('#^data:#i', $url);
	}

	public static function isInvalid($url)
	{
		return (empty($url) || trim($url) == '/');
	}

	public static function isHttpScheme($url)
	{
		return !preg_match('#^(?!https?)[^:/]+:#i', $url);
	}

	public static function AbsToProtocolRelative($url)
	{
		return preg_replace('#https?:#i', '', $url);
	}

	public static function toRootRelative($url, $file = '')
	{
		if (self::isPathRelative($url))
		{
			$url = (empty($file) ? '' : dirname($file) . '/' ) . $url;
		}

		$url = self::getInstance($url)->toString(array('path', 'query', 'fragment'));

		if (self::isPathRelative($url))
		{
			$url = rtrim(JUri::root(true), '\\/') . '/' . $url;
		}

		return $url;
	}

	public static function toAbsolute($url, $file)
	{
		$uriObject = clone self::getInstance($file);

		if (self::isPathRelative($url))
		{
			$uriObject->setPath(dirname($uriObject->getPath()) . '/' . $url);
		}

		if (self::isRootRelative($url))
		{
			$uriObject->setPath($url);
		}

		if (self::isProtocolRelative($url))
		{
			$scheme = $uriObject->getScheme();

			if (!empty($scheme))
			{
				$url = $scheme . ':' . $url;
			}

			$uriObject = self::getInstance($url);
		}

		$url = $uriObject->toString();
		$host = $uriObject->getHost();

		if (!self::isAbsolute($url) && !empty($host)) 
		{
			return '//' .  $url;
		}

		return $url;
	}

	public static function requiresHttpProtocol($url)
	{
		return preg_match('#\.php|^(?![^?\#]*\.(?:css|js|png|jpe?g|gif|bmp)(?:[?\#]|$)).++#i', $url);
	}
	
	public static function path2Url($path)
	{
		$uri = clone JUri::getInstance();
		
		$url  = $uri->toString(array('scheme', 'user', 'pass', 'host', 'port'));
		$url .= JUri::root(true) . '/';
		$url .= self::strReplace(JPATH_ROOT . DIRECTORY_SEPARATOR, '', $path);

		return $url;
	}
	
	public static function strReplace($search, $replace, $subject)
	{
		return str_replace(self::clean($search), $replace, self::clean($subject));
	}
	
	public static function clean($url)
	{
		return str_replace(array('\\\\', '\\'), '/', $url);
	}
	
	public static function getFilePath($url)
	{
		$base = JUri::root(true);
		
		$uri = clone self::getInstance();
		$url_object = clone self::getInstance(html_entity_decode($url));

		// Use absolute file path if file is internal and a static file
		if (self::isInternal($url) && !self::requiresHttpProtocol($url))
		{
			$absolute_path = $url_object->getPath();

			if (!empty($base) && strpos($absolute_path, $base) === 0)
			{
				$absolute_path = substr($absolute_path, strlen($base));
			}

			return self::absolutePath($absolute_path);
		}
		else
		{
			$scheme = $url_object->getScheme();

			if (empty($scheme))
			{
				$url_object->setScheme($uri->getScheme());
			}

			$host = $url_object->getHost();

			if (empty($host))
			{
				$url_object->setHost($uri->getHost());
			}

			$path = $url_object->getPath();

			if (!empty($path))
			{
				if (substr($path, 0, 1) != '/')
				{
					$url_object->setPath($sUriPath . '/' . $path);
				}
			}

			$url   = $url_object->toString();
			$query = $url_object->getQuery();

			if (!empty($query))
			{
					parse_str($query, $args);

					$url = str_replace($query, http_build_query($args, '', '&'), $url);
			}

			return $url;
		}
	}
	
	public static function absolutePath($url)
	{
		return JPATH_ROOT . DIRECTORY_SEPARATOR . ltrim(str_replace('/', DIRECTORY_SEPARATOR, $url), '\\/');
	}
}
