<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

/**
* VP Framework Utilities
* 
* @since 1.0.0
*/
class VPFrameworkUtility
{
	private static $extensions = array();
	
	public static function getRemoteData($url)
	{
		$useragent = "Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:25.0) Gecko/20100101 Firefox/25.0";
		$data = false;

		// cURL
		if (extension_loaded('curl'))
		{
			// Init cURL
			$ch = @curl_init();
			
			// Set options
			@curl_setopt($ch, CURLOPT_URL, $url);
			@curl_setopt($ch, CURLOPT_HEADER, 0);
			@curl_setopt($ch, CURLOPT_FAILONERROR, 1);
			@curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
			@curl_setopt($ch, CURLOPT_USERAGENT, $useragent);
			
			// Set timeout
			@curl_setopt($ch, CURLOPT_TIMEOUT, 5);
			
			// Grab data
			$data = @curl_exec($ch);
			
			// Clean up
			@curl_close($ch);

			// Return data
			if ($data !== false)
			{
				return $data;
			}
		}

		// fsockopen
		if (function_exists('fsockopen'))
		{
			$errno = 0;
			$errstr = '';

			$url_info = parse_url($url);
			if ($url_info['host'] == 'localhost')
			{
				$url_info['host'] = '127.0.0.1';
			}

			// Set timeout
			$fsock = @fsockopen($url_info['scheme'].'://'.$url_info['host'], 80, $errno, $errstr, 5);

			if ($fsock)
			{
				@fputs($fsock, 'GET ' . $url_info['path'].(!empty($url_info['query']) ? '?' . $url_info['query'] : '') . ' HTTP/1.1'."\r\n");
				@fputs($fsock, 'HOST: '.$url_info['host'] . "\r\n");
				@fputs($fsock, "User-Agent: " . $useragent . "\n");
				@fputs($fsock, 'Connection: close' . "\r\n\r\n");

				// Set timeout
				@stream_set_blocking($fsock, 1);
				@stream_set_timeout($fsock, 5);

				$data = '';
				$passed_header = false;
				while (!@feof($fsock))
				{
					if ($passed_header)
					{
						$data .= @fread($fsock, 1024);
					}
					else
					{
						if (@fgets($fsock, 1024) == "\r\n")
						$passed_header = true;
					}
				}

				// Clean up
				@fclose($fsock);

				// Return data
				if ($data !== false)
				{
					return $data;
				}
			}
		}

		// fopen
		if (function_exists('fopen') && function_exists('ini_get') && ini_get('allow_url_fopen'))
		{
			// Set timeout
			if (ini_get('default_socket_timeout') < 5)
			{
				ini_set('default_socket_timeout', 5);
			}
			
			@stream_set_blocking($handle, 1);
			@stream_set_timeout($handle, 5);
			@ini_set('user_agent',$useragent);

			$url = str_replace('://localhost', '://127.0.0.1', $url);

			$handle = @fopen ($url, 'r');

			if ($handle)
			{
				$data = '';
				while (!feof($handle))
				{
					$data .= @fread($handle, 8192);
				}

				// Clean up
				@fclose($handle);

				// Return data
				if ($data !== false)
				{
					return $data;
				}
			}
		}

		// file_get_contents
		if (function_exists('file_get_contents'))
		{
			$url = str_replace('://localhost', '://127.0.0.1', $url);
			if (function_exists('ini_get'))
			{
				@ini_set('user_agent', $useragent);
			}
			$data = @file_get_contents($url);

			// Return data
			if ($data !== false)
			{
				return $data;
			}
		}

		return $data;
	}
	
	public static function getMessages()
	{
		$messages = '';
		$msgs = JFactory::getApplication()->getMessageQueue();
		
		if (!empty($msgs))
		{
			$messages = array();
			foreach ($msgs as $msg) 
			{
				$message = str_replace(array('<br/>', '<br />', '<BR/>'), array('||*||'), $msg['message']);
				
				if (strpos($message, '||*||') !== false)
				{
					$message = explode('||*||', $message);
					$messages = array_merge($messages, $message);
				}
				else
				{
					$messages[] = $msg['message'];
				}
			}
			
			if (!empty($messages))
			{
				$messages = array_map('trim', $messages);
				$messages = array_unique(array_filter($messages));
				$messages = implode('<br/>', $messages);
			}
			else
			{
				$messages = '';
			}
		}

		return $messages;
	}
	
	
	public static function jsonReturn($output = array()) 
	{
		$app = JFactory::getApplication();
		$obLevel = ob_get_level();
		
		if ($obLevel)
		{
			while($obLevel > 0)
			{
				ob_end_clean();
				$obLevel --;
			}
		}
		elseif (ob_get_contents())
		{
			ob_clean();
		}
		
		header('Content-type: application/json');
		header('Cache-Control: public,max-age=1,must-revalidate');
		header('Expires: ' . gmdate('D, d M Y H:i:s', ($_SERVER['REQUEST_TIME'] + 1)) . ' GMT');
		header('Last-modified: ' . gmdate('D, d M Y H:i:s', $_SERVER['REQUEST_TIME']) . ' GMT');
			
		if (function_exists('header_remove')) 
		{
			header_remove('Pragma');
		}
						
		echo json_encode($output);
		flush();
		$app->close();
	}
	
	
	public static function hasExtension($element, $type = 'component')
	{
		jimport('joomla.filesystem.folder');
		
		if (!array_key_exists($element, self::$extensions))
		{
			$type = strtolower($type);
			self::$extensions[$element] = false;
			
			$db = JFactory::getDbo();
			$query = $db->getQuery(true)
			            ->select('element, folder')
			            ->from('#__extensions')
			            ->where('type = ' . $db->quote($type))
			            ->where('element = ' . $db->quote($element))
			            ->where('enabled = 1');
			try
			{
				$db->setQuery($query);
				$result = $db->loadObject();
			}
			catch(Exception $e)
			{
				JFactory::getApplication()->enqueueMessage($e->getMessage(), 'error');
				return false;
			}
			
			if (!empty($result))
			{
				switch($type)
				{
					case 'module':
						$path = JPATH_SITE . '/modules/' . $result->element;
						break;
					case 'plugin':
						$path = JPATH_SITE . '/plugins/' . $result->folder . '/' . $result->element;
						break;
					default:
					case 'component':
						$path = JPATH_ADMINISTRATOR . '/components/' . $result->element;
				}
				
				if (JFolder::exists($path))
				{
					self::$extensions[$element] = true;
				}
			}
		}
		
		return self::$extensions[$element];
	}
	
	/**
	* strpos() where needles is array
	* 
	* @param string  $haystack  Haystack to check
	* @param array   $needles   Array of needles
	* @param integer $offset    Offset in the haystack
	* @param boolean $getNeedle Default should be 'false'. If the a call finds a match then you can retreve the matched 
	*                                                      needle by calling it again setting this value 'true'.
	* 
	* @return mixed (boolean/integer/string) False if not found or integer value of found position or found needle (string)
	*/
	public static function strposa($haystack, $needles, $offset = 0, $getNeedle = false)
	{
		static $foundNeedles = array();
		
		$key = (string) $haystack . serialize($needles) . $offset;
		$foundNeedles[$key] = null;
		
		if ($getNeedle)
		{
			$foundNeedle = isset($foundNeedles[$key]) ? $foundNeedles[$key] : '';
			return $foundNeedle;
		}
		
		if (isset($foundNeedles[$key]))
		{
			return $foundNeedles[$key];
		}
		
		foreach ($needles as $needle)
		{
			$pos = strpos($haystack, $needle, $offset);
			
			if ($pos !== false)
			{
				$foundNeedles[$key] = $needle;
				return $pos;
			}
		}
		
		return false;
	}
	
	public static function getLast($array, $getKey = false)
	{
		if (empty($array))
		{
			if ($getKey)
			{
				return 0;
			}
			
			return $array;
		}
		
		if ($getKey)
		{
			end($array);
			
			return key($array);
		}
		
		return end($array);
	}
	
	public static function sortArrayByArray(array $array, array $orderArray)
	{
		$ordered = array();
		
		foreach ($orderArray as $key)
		{
			if (array_key_exists($key, $array))
			{
				$ordered[$key] = $array[$key];
				unset($array[$key]);
			}
		}
		
		return $ordered + $array;
	}

	public static function sanitizeOutput($buffer)
	{
		$search = array(
		    '/\>[^\S ]+/s',  // strip whitespaces after tags, except space
		    '/[^\S ]+\</s',  // strip whitespaces before tags, except space
		    '/(\s)+/s'       // shorten multiple whitespace sequences
		);

		$replace = array(
		    '>',
		    '<',
		    '\\1'
		);

		$buffer = preg_replace($search, $replace, $buffer);

		return $buffer;
	}
}