<?php
/***
 * @package customfilters
 * @author Sakis Terz
 * @copyright Copyright (C) 2012-2023 breakdesigns.net . All rights reserved.
 * @license GNU General Public License version 2 or later; see LICENSE.txt
 */

use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die();

/**
 * Used to fetch attributes (e.g. name,alias) of custom filters.
 *
 * @since       2.15.0
 */
class CustomFieldHelper extends CfItemsHelper
{
    /**
     * @var stdClass
     * @since 2.15.0
     */
    protected $customFilter;

    public function __construct(\stdClass $customFilter)
    {
        $this->customFilter = $customFilter;
    }

	/**
	 * Get the items
	 *
	 * @param array  $ids
	 * @param string $valuesType
	 *
	 * @return array|bool
	 *
	 * @since version
	 */
	public function getItems(array $ids, string $valuesType = 'ids')
	{
		if ($this->customFilter->field_type != 'E') {
			// Sort alphabetically
			$ids = cftools::sort_by('', $ids);
		}

		$items = parent::getItems($ids, $valuesType);
        $firstItem = reset($items);

		// Sort plugin values by their ordering field
		if ($items && isset($firstItem->ordering)) {
			$items = cftools::sort_by('ordering', $items);
		}

		return $items;
	}

    /**
     * B/C function.
     * Seems that VMUkit is overriding our view and calls old functions.
     *
     * @param int $custom_id
     * @param $ids
     *
     * @return array|bool|string
     * @since 2.14.0
     */
    public static function getNames(...$params)
    {
        $custom_id = $params[0];
        $ids = $params[1];
        $customFieldsHelper = CustomFieldsHelper::getInstance();
        return $customFieldsHelper->getCustomField($custom_id)->get('name', $ids);
    }

    /**
	 * Get the items from the database
	 *
	 * @param array   $values
	 * @param string  $valuesType
	 * @param bool    $valuesHex
	 *
	 * @return array|void
	 * @since 2.15.0
	 */
    protected function getItemsFromStorage(array $values, $valuesType = 'ids', $valuesHex = false)
    {
        $items = [];

		if ($valuesHex) {
			$values = cftools::hex2binArray($values);
		}

        // is plugin
        if ($this->customFilter->field_type == 'E' &&
            isset($this->customFilter->pluginparams) &&
            $this->customFilter->pluginparams->product_customvalues_table != $this->customFilter->pluginparams->customvalues_table &&
            strpos($this->customFilter->disp_type, CfFilter::DISPLAY_INPUT_TEXT) === false &&
            strpos($this->customFilter->disp_type, CfFilter::DISPLAY_RANGE_SLIDER) === false &&
            strpos($this->customFilter->disp_type, CfFilter::DISPLAY_RANGE_DATES) === false)
        {
            $pluginparams = $this->customFilter->pluginparams;
            $db = Factory::getDbo();
            $query = $db->getQuery(true);
            $valuesQuoted = array_map([$db, 'quote'], $values);

            // From can be a single table or a sub-query (e.g. multi-lingual custom fields).
            $fromClause = strrpos($pluginparams->customvalues_table, '#__') == 0 ? $db->quoteName($pluginparams->customvalues_table, 'cf') : $pluginparams->customvalues_table . ' AS cf ';
			$query->select(
                [
                    $db->quoteName($pluginparams->filter_by_field, 'id'),
                    $db->quoteName($pluginparams->customvalue_value_field, 'name'),
                    $db->quoteName($pluginparams->customvalue_value_field, 'alias'),
                    $db->quoteName($pluginparams->customvalue_value_description_field, 'description')
                ])
                  ->from($fromClause);
			if ($valuesType == 'ids') {
				$query->where($db->quoteName($pluginparams->filter_by_field) . ' IN(' . implode(',', $valuesQuoted) . ')');
			} else {
				// In case of colors the values can be either the custom field id, or the description/label
				if (isset($this->customFilter->data_type) && $this->customFilter->data_type == 'color_hex') {
					$valueIds = [];
					$valueLabels = [];
					foreach ($values as $value)
					{
						if (preg_match('/^\d+$/', $value))
						{
							$valueIds[] = $value;
						} else {
							$valueLabels[] = $value;
						}
					}
					$valueLabels = array_map([$db, 'quote'], $valueLabels);
					$valueIds = array_map([$db, 'quote'], $valueIds);
					$where = [];
					if ($valueLabels) {
						$where[]= $db->quoteName($pluginparams->customvalue_value_description_field) . ' IN(' . implode(',', $valueLabels) . ')';
					}
					if ($valueIds) {
						$where[]= $db->quoteName($pluginparams->filter_by_field) . ' IN(' . implode(',', $valueIds) . ')';
					}

					$query->where('(' . implode(' OR ', $where) .')');
				}
				else {
					$query->where($db->quoteName($pluginparams->customvalue_value_field) . ' IN(' . implode(',', $valuesQuoted) . ')');
				}

				$query->where($db->quoteName('virtuemart_custom_id') . '=' . $db->quote($this->customFilter->custom_id));
			}

			if ($pluginparams->sort_by) {
				$query->select($db->quoteName($pluginparams->sort_by, 'ordering'));
			}
            $db->setQuery($query);
            $items = $db->loadObjectList();

            // If it's a color check if it has label/description which can be used instead of it's plain value
            if ($items && isset($this->customFilter->data_type) && $this->customFilter->data_type == 'color_hex' && $valuesType == 'ids') {
                foreach ($items as $item) {
	                $item->alias = $item->id;
                    if (!empty($item->description)) {
                        $item->name = $item->description;
	                    $item->alias = $item->description;
                    }
                }
            }

            $newItems = [];
            foreach ($items as $item) {
                $key = in_array($item->id, $values) ? $item->id : (in_array($item->name, $values) ? $item->name : $item->description);
				$key = $valuesHex ? bin2hex($key) : $key;
                $newItems[$key] = $item;
            }

            $items = $newItems;

        } // If not plugin or a plugin that uses no other table for storing the values
        elseif ($this->customFilter->field_type != 'E' || isset($this->customFilter->pluginparams)) {

            // Use the provided plain values
            foreach ($values as $value) {
                $item = new \stdClass();
                $item->name = $value;
                $item->alias = $value;
                $items[bin2hex($value)] = $item;
            }
        }

        return $items;
    }
}