/*
Class: Slider
        Creates a slider with two elements: a knob and a container. Returns the values.
Arguments:
        key - the elements selector prefix
        module_id - the module id
        options - see Options below
Options:
		start - the minimum value for your slider.
		end - the maximum value for your slider.
		listen - calls the customFilters.listen() which triggers ajax requests.
		knobHeight
		knobWidth
		minDelta - The min. difference that min and max should have
*/
class Cfslider {
    constructor(key, module_id, options) {
        this.options = {
            start: 0,
            end: 1000,
            listen: false,
            knobHeight: 20,
            knobWidth: 20,
            minDelta: 1
        }

        const el = key + '_slider_gutter_m';
        const minKnob = key + '_knob_from';
        const maxKnob = key + '_knob_to';
        const bkg = key + '_slider_bkg';

        this.key = key;
        this.module_id = module_id;
        this.setOptions(options);
        this.element = document.getElementById(el);
        this.minKnob = document.getElementById(minKnob);
        this.maxKnob = document.getElementById(maxKnob);
        this.bkg = document.getElementById(bkg);
        this.steps = parseInt(this.options.end) - parseInt(this.options.start);
        this.sliderWidth = this.minKnob.clientWidth;

        this.updateTooltip('tipFrom', this.minKnob);
        this.updateTooltip('tipTo', this.maxKnob);
        this.init();
    }

    setOptions(options) {
        for (const [key, value] of Object.entries(options)) {
            this.options[key] = value;
        }
    }

    init() {
        // Remove the module id from the filter key
        let filterName = this.key.replace(/(\d+)$/, '');
        // Remove the ending undescrore too
        filterName = filterName.substring(0, filterName.length - 1);

        this.minKnob.addEventListener('input', () => {
            const minInput = document.getElementById(this.key + '_0');
            const minValue = parseInt(this.minKnob.value);
            const maxValue = parseInt(this.maxKnob.value);

            //If the lower value slider is GREATER THAN the upper value slider minus one.
            if (minValue > maxValue - this.options.minDelta) {
                //The upper slider value is set to equal the lower value slider plus one.
                this.maxKnob.value = minValue + this.options.minDelta;
                //If the upper value slider equals its set maximum.
                if (maxValue == this.maxKnob.max) {
                    //Set the lower slider value to equal the upper value slider's maximum value minus one.
                    this.minKnob.value = parseInt(this.maxKnob.max) - this.options.minDelta;
                }

                const maxInput = document.getElementById(this.key + '_1');
                if (maxInput) {
                    maxInput.value = this.maxKnob.value;
                }
            }
            if (minInput) {
                minInput.value = this.minKnob.value;
            }

            // Update the tooltip
            this.updateTooltip('tipFrom', this.minKnob);
            this.fillRange();
        })

        // Triggered when the slide is finished
        this.minKnob.addEventListener('change', () => {
            if (this.options.listen && typeof customFilters.listen == 'function') {
                customFilters.listen(null, this.element, this.module_id);
            }else if(typeof customFilters.validateRangeFlt == 'function') {
                customFilters.validateRangeFlt(this.module_id, filterName);
            }
        })

        this.maxKnob.addEventListener('input', () => {
            const maxInput = document.getElementById(this.key + '_1');
            const minValue = parseInt(this.minKnob.value);
            const maxValue = parseInt(this.maxKnob.value);

            //If the upper value slider is LESS THAN the lower value slider plus one.
            if (maxValue < minValue + this.options.minDelta) {
                //The lower slider value is set to equal the upper value slider minus one.
                this.minKnob.value = maxValue - this.options.minDelta;
                //If the lower value slider equals its set minimum.
                if (minValue == this.minKnob.min) {
                    //Set the upper slider value to equal 1.
                    this.maxKnob.value = this.options.minDelta;
                }
                const minInput = document.getElementById(this.key + '_0');
                if (minInput) {
                    minInput.value = this.minKnob.value;
                }
            }
            if (maxInput) {
                maxInput.value = this.maxKnob.value;
            }

            // Update the tooltip
            this.updateTooltip('tipTo', this.maxKnob);
            this.fillRange();
        })

        // Triggered when the slide is finished
        this.maxKnob.addEventListener('change', () => {
            if (this.options.listen && typeof customFilters.listen == 'function') {
                customFilters.listen(null, this.element, this.module_id);
            }else if(typeof customFilters.validateRangeFlt == 'function') {
                customFilters.validateRangeFlt(this.module_id, filterName);
            }
        })
    }

    setMin(minValue) {
        if (parseInt(minValue) + this.options.minDelta <= parseInt(this.maxKnob.value)) {
            this.minKnob.value = minValue;
            this.fillRange();
        }
        return this;
    }

    setMax(maxValue) {
        if (parseInt(maxValue) - this.options.minDelta >= parseInt(this.minKnob.value)) {
            this.maxKnob.value = maxValue;
            this.fillRange();
        }
        return this;
    }

    fillRange() {
        const minValue = parseInt(this.minKnob.value);
        const minLimit = parseInt(this.minKnob.min);
        const maxValue = parseInt(this.maxKnob.value);
        // Find px/step
        const ratio = this.sliderWidth / this.steps;
        const left = ratio * (minValue - minLimit);
        //Setting the margin left of the middle range color.
        this.bkg.style.marginLeft = left + "px";
        const right = ratio * (maxValue - minLimit);
        //Setting the width of the middle range color.
        this.bkg.style.width = right - left + "px";
    }

    updateTooltip(tip, rangeElement) {
        if(typeof window[tip+this.key] != 'undefined') {
            window[tip+this.key].setValue(rangeElement.value);
            window[tip+this.key].setCharLength(rangeElement.value.length);

            // Find px/step
            const ratio = this.sliderWidth / this.steps;
            const minLimit = parseInt(this.minKnob.min);
            const xPos = ratio * (parseInt(rangeElement.value) - minLimit);

            window[tip+this.key].positionX(xPos);
        }
    }
}