<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

defined('PHP_EOL') or define('PHP_EOL', "\n");

class VPFrameworkDlk extends JObject
{
	public $name;
	public $id;
	public $value;
	public $framework;
	public $params;
	public $package;
	public $template;
	public $pid;
	
	protected $input;
	protected $adapter;
	protected $time_start;
	protected $time_end;
	protected $remote_host;
	protected $remote_host_secured;
	protected $try_count;
	
	protected static $instance      = null;
	protected static $packages      = null;
	protected static $update_server = 'www.virtueplanet.com';
	
	
	public function __construct($params = null)
	{
		jimport('joomla.http.factory');

		if (!class_exists('JHttpFactory'))
		{
			throw new BadFunctionCallException('JHttpFactory not present. Please upgrade your version of Joomla. Exiting optimizer.');
		}

		$this->time_start = microtime(true);
		
		$app       = JFactory::getApplication();
		$framework = JPluginHelper::getPlugin('system', 'vpframework');
		$params    = new JRegistry;
		
		if (!empty($framework->params))
		{
			if (is_string($framework->params))
			{
				$params->loadString($framework->params);
			}
			elseif (is_array($framework->params))
			{
				$params->loadArray($framework->params);
			}
			elseif (is_object($framework->params))
			{
				$params = $framework->params;
			}
		}
		
		$this->framework  = $framework;
		$this->params     = $params;
		$this->name       = 'vpf_dlk';
		$this->id         = 'vpf-dlk';
		$options          = new JRegistry(array('follow_location' => true));
		$curl_opts        = array();
		
		if (defined('CURLOPT_SSL_VERIFYHOST'))
		{
			$curl_opts[CURLOPT_SSL_VERIFYHOST] = false;
		}
		
		if (defined('CURLOPT_SSL_VERIFYPEER'))
		{
			$curl_opts[CURLOPT_SSL_VERIFYPEER] = false;
		}
		
		if (!empty($curl_opts))
		{
			$options->set('transport.curl', $curl_opts);
		}

		$this->adapter = JHttpFactory::getAvailableDriver($options);
		$this->input   = $app->input;
		
		// Load other necessary information
		$this->loadInfo();
		
		if (is_object($this->package) && !empty($this->package->extension_id))
		{
			$extension_id = (int) $this->package->extension_id;
			$dlks         = (array) $params->get('dlks', array());
			$key          = 'pack_id_' . $extension_id;
			$this->value  = is_array($dlks) && isset($dlks[$key]) ? $dlks[$key] : '';
		}
		
		$this->remote_host = 'http://validate.virtueplanet.com';
		$this->try_count   = 0;
		
		if (function_exists('extension_loaded') && extension_loaded('openssl'))
		{
			// Secured verification over SSL
			$this->remote_host_secured = 'https://www.virtueplanet.com/validate';
		}
	}

	/**
	* Method to get an instance of the the VVPFrameworkDlk class
	* 
	* @return object VPFrameworkDlk class object
	*/
	public static function getInstance()
	{
		if (self::$instance === null)
		{
			self::$instance = new VPFrameworkDlk();
		}
		
		return self::$instance;
	}
	
	
	public function getInput()
	{
		$doc        = JFactory::getDocument();
		$app        = JFactory::getApplication();
		$today      = JFactory::getDate();
		$rel_path   = str_replace(JPATH_ROOT, '', dirname(__FILE__));
		$root       = JUri::root(true) . '/';
		$base_url   = $root . ltrim(str_replace('\\', '/', $rel_path), '/');
		$assets_url = $base_url . '/dlk/assets';
		$html       = array();
		
		JHtml::_('jquery.framework');
		
		// Load CSS
		$doc->addStyleSheet($assets_url . '/css/style.css');
		
		// Load JS
		$doc->addScript($assets_url . '/js/script.min.js');
		
		$this->value  = empty($this->value) && empty($this->adapter) ? 'dummy' : $this->value;
		$data         = self::decodeData($this->value);
		$last_checked = $data['last_checked'] ? JFactory::getDate($data['last_checked']) : $today;
		$required     = ' required="required"';
		$btn_class    = !empty($data['dlk']) ? ' vpdk-show-edit' : '';
		$clear_class  = !empty($data['dlk']) ? '' : ' vpdk-hide';
		$interval     = date_diff($last_checked, $today);
		$interval     = (int) $interval->format('%R%a'); // Interval in days
		$revalidate   = ($last_checked != $today) && ($interval > 7) ? ' data-vpdkvalidate="auto"' : '';
		$access_class = !empty($data['dlk']) && !$data['access'] ? '' : ' vpdk-hide';
		$valid_class  = !empty($data['dlk']) && $data['access'] ? '' : ' vpdk-hide';
		$def_class    = empty($data['dlk']) ? '' : ' vpdk-hide';
		$data_access  = !$data['access'] ? ' data-vpdkaccess="0"' : ' data-vpdkaccess="1"';

		$html[] = '<div class="control-group">';
		$html[] = '<div class="control-label">';
		$html[] = '<label id="' . $this->id . '-lbl" for="' . $this->id . '" class="required">';
		$html[] = 'Download Key<span class="star">&nbsp;*</span>';
		$html[] = '</label>';
		$html[] = '</div>';
		$html[] = '<div class="controls">';

		if (version_compare(JVERSION, '4.0.0', 'ge'))
		{
			$html[] = '<div class="input-group">';
		}

		$html[] = '<input type="password" id="' . $this->id . '" value="'
		          . htmlspecialchars($data['dlk'], ENT_COMPAT, 'UTF-8') . '" data-vpdk="validate"' . $required . $data_access 
		          . ' autocomplete="off" spellcheck="false" class="form-control" placeholder="Download Key" readonly />';
		
		if (empty($this->adapter))
		{
			if (version_compare(JVERSION, '4.0.0', 'ge'))
			{
				// Close input group
				$html[] = '</div>';
			}
			
			// Close controls
			$html[] = '</div>';

			JError::raiseWarning(100, 'JHttpFactory not present. Please upgrade your version of Joomla.');
		}
		else
		{
			if (version_compare(JVERSION, '4.0.0', 'ge'))
			{
				$html[] = '<button type="button" id="' . $this->id . '-button" class="btn btn-outline-secondary' . $btn_class . '" data-vpdktarget="' . $this->id . '-tmpl-modal">';
				$html[] = '<span class="vpdk-add-text">Add</span>';
				$html[] = '<span class="vpdk-edit-text">Edit</span>';
				$html[] = '</button>';
				$html[] = '<button type="button" id="' . $this->id . '-clear-button" class="btn btn-outline-danger' . $clear_class . '">Clear</button>';
				$html[] = '<button type="button" id="' . $this->id . '-reval-button" class="btn btn-outline-info vpdk-reval-button' . $clear_class . '"' . $revalidate . '>Revalidate<span class="vpk-btn-overlay"></span></button>';
				
				// Close input group
				$html[] = '</div>';
			}
			else
			{
				$html[] = '<button type="button" id="' . $this->id . '-button" class="pure-button' . $btn_class . '" data-vpdktarget="' . $this->id . '-tmpl-modal">';
				$html[] = '<span class="vpdk-add-text">Add</span>';
				$html[] = '<span class="vpdk-edit-text">Edit</span>';
				$html[] = '</button>';
				$html[] = '<button type="button" id="' . $this->id . '-clear-button" class="pure-button' . $clear_class . '">Clear</button>';
				$html[] = '<button type="button" id="' . $this->id . '-reval-button" class="pure-button blue vpdk-reval-button' . $clear_class . '"' . $revalidate . '>Revalidate<span class="vpk-btn-overlay"></span></button>';
			}

			$html[] = '<div class="vpdk-info-box">';
			$html[] = '<div class="vpdk-info-noaccess' . $access_class . '">';
			$html[] = '<p>You don\'t have access to the latest updates for this ' . $this->input->getCmd('view', 'extension') . '. Your subscription may have expired or you may not have a valid subscription for this item. To learn more about your subscription plan please visit your <a href="" target="_blank">Dashboard</a>.</p>';
			
			if ($last_checked != $today)
			{
				$html[] = '<small>Last validated on: <span data-lastchecked="date">' . JHtml::_('date', $last_checked, 'F d, Y H:i:s') . '</span></small>';
			}
			
			$html[] = '</div>';
			$html[] = '<div class="vpdk-info-valid' . $valid_class . '">';
			$html[] = '<p>Your subscription is active and you also have access to live updates.</p>';
			
			if ($last_checked != $today)
			{
				$html[] = '<small>Last validated on: <span data-lastchecked="date">' . JHtml::_('date', $last_checked, 'F d, Y H:i:s') . '</span></small>';
			}
			
			$html[] = '</div>';
			$html[] = '<div class="vpdk-info-default' . $def_class . '">';
			$html[] = '<p>Please add your Download Key above. <a href="https://www.virtueplanet.com/download-key/" target="_blank">Forgot Download Key?</a></p>';
			$html[] = '</div>';
			$html[] = '</div>';
			$html[] = $this->getFormTemplate($this->id . '-tmpl');
			
			// Close controls
			$html[] = '</div>';
		}

		return implode(PHP_EOL, $html);
	}
	
	protected function getFormTemplate($id, $html = array())
	{
		$extension_id = is_object($this->package) && !empty($this->package->extension_id) ? (int) $this->package->extension_id : 0;
		$template_id  = $this->input->getInt('id', 0);
		$url          = JRoute::_('index.php?option=com_templates&view=style&layout=edit&id=' . $template_id . '&extension_id=' . $extension_id . '&' . JSession::getFormToken() . '=1');
		
		$html[] = '<script id="' . $id . '" type="text/x-jquery-tmpl">';
		$html[] = '<div id="' . $id . '-modal" class="vpdk-modal">';
		$html[] = '<div class="vpdk-modal-dialog">';
		$html[] = '<div class="vpdk-modal-inner">';
		$html[] = '<form data-action="' . $url . '">';
		$html[] = '<div class="vpdk-modal-header">';
		$html[] = '<button type="button" class="vpdk-modal-close" title="Close">';
		$html[] = '<span aria-hidden="true">&times;</span>';
		$html[] = '</button>';
		$html[] = '<h4>Validate &amp Add Download Key</h4>';
		$html[] = '</div>';
		$html[] = '<div class="vpdk-modal-body">';
		$html[] = '<div class="vpdk-modal-message"></div>';
		$html[] = '<div class="vpdk-form-group">';
		$html[] = '<label for="' . $id . '-uname">VirtuePlanet Username</label>';
		$html[] = '<input type="text" id="' . $id . '-uname" name="uname" class="vpdk-form-control" placeholder="Username" autocomplete="off" spellcheck="false" required />';
		$html[] = '</div>';
		$html[] = '<div class="vpdk-form-group">';
		$html[] = '<label for="' . $id . '-dlk">Download Key</label>';
		$html[] = '<input type="text" id="' . $id . '-dlk" name="dlk" class="vpdk-form-control" placeholder="Download Key" autocomplete="off" spellcheck="false" required />';
		$html[] = '</div>';
		$html[] = '<div class="vpdk-modal-buttons">';
		$html[] = '<button type="button" id="' . $id . '-button" class="vpdk-btn vpdk-btn-lg vpdk-btn-primary"><i class="vpdk-button-loading"></i>Submit</button>';
		$html[] = '<button type="reset" id="' . $id . '-cancel" class="vpdk-btn vpdk-btn-lg vpdk-btn-default">Cancel</button>';
		$html[] = '</div>';
		$html[] = '</div>';
		$html[] = '<div class="vpdk-modal-footer">';
		$html[] = '<ul>';
		$html[] = '<li><a href="https://www.virtueplanet.com/download-key/" target="_blank">Forgot Download Key?</a></li>';
		$html[] = '<li><a href="https://www.virtueplanet.com/lost-user-name/" target="_blank">Forgot Username?</a></li>';
		$html[] = '</ul>';
		$html[] = '</div>';
		$html[] = '</form>';
		$html[] = '</div>';
		$html[] = '<div class="vpdk-modal-inner-bg"></div>';
		$html[] = '</div>';
		$html[] = '</div>';
		$html[] = '</script>';
		
		return implode(PHP_EOL, $html);
	}
	
	public function clear()
	{
		$result = array('error' => false, 'msg' => '');
		
		if (!$this->save('***'))
		{
			$result['error'] = true;
			$result['msg'] = $this->getError();
		}
		
		if (!$result['error'])
		{
			if (!$this->refreshUpdates('', true))
			{
				$result['error'] = true;
				$result['msg'] = $this->getError();
			}
		}
		
		return $this->doReturn($result);
	}
	
	public function validate()
	{
		$query = array();
		
		$query['uname']  = $this->input->get('uname', '', 'GET', 'USERNAME');
		$dlk             = $this->input->get('dlk', '', 'GET', 'STRING');
		$query['dlk']    = self::cleanDlk($dlk);
		$query['pid']    = (int) $this->pid;
		$query['host']   = base64_encode(self::getHost());
		$query['_pkey']  = md5('time:' . $this->time_start . '.rand:' . mt_rand());
		
		return $this->_validate($query);
	}
	
	public function revalidate()
	{
		$data  = self::decodeData($this->value, true);
		$query = array();
		
		$query['uname']  = $data['uname'];
		$query['dlk']    = $data['dlk'];
		$query['pid']    = (int) $this->pid;
		$query['host']   = base64_encode(self::getHost());
		$query['_pkey']  = md5('time:' . $this->time_start . '.rand:' . mt_rand());
		
		return $this->_validate($query);
	}
	
	public static function addDlk(&$url, &$headers)
	{
		if (strpos($url, 'https://' . self::$update_server) !== 0 && strpos($url, 'http://' . self::$update_server) !== 0)
		{
			return true;
		}
		
		// If dlid is already added we do not need to do anything
		if (strpos($url, 'dlid=') !== false)
		{
			return true;
		}
		
		// Get all VP Framework based template packages who has dlk
		$packages = self::getPackages();
		
		if (!count($packages))
		{
			return true;
		}
		
		$package_names = array_keys($packages);
		$url_lower     = strtolower($url);
		
		foreach ($package_names as $package_name)
		{
			if (!is_string($package_name))
			{
				continue;
			}
			
			$name = strtolower($package_name);
			
			if (strpos($url_lower, $name) === false)
			{
				continue;
			}
			
			if (!isset($packages[$package_name]))
			{
				continue;
			}
			
			$package = $packages[$package_name];
			
			if (strpos($url, '?') === false)
			{
				$url .= '?dlid=' . $package['dlk'];
			}
			else
			{
				$url .= '&dlid=' . $package['dlk'];
			}
			
			return true;
		}

		return true;
	}
	
	protected function _validate($query)
	{
		$this->try_count++;
		
		$result = array('error' => true, 'msg' => '', 'return' => null, 'adapter' => '');
		
		if (!JSession::checkToken('get'))
		{
			return $this->doReturn($result);
		}
		
		if (!$this->adapter)
		{
			$result['msg'] = 'JHttpFactory not present. Please upgrade your version of Joomla.';
			
			return $this->doReturn($result);
		}
		
		$url = $this->remote_host . '/index.php';
		$uri = JUri::getInstance($url);
		$userAgent = !empty($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'VirtuePlanet Verifier/2.0';
		$errorMessage = null;
		
		try
		{
			$data = http_build_query($query, '', '&');
			$response = $this->adapter->request('POST', $uri, $data, array(), 30, $userAgent);
		}
		catch(Exception $e)
		{
			$response = new stdClass;
			
			$response->code = 200;
			$response->body = ' .well-known captcha';
			
			$errorMessage = $e->getMessage();
		}
		
		if (empty($response))
		{
			$result['msg'] = 'Verification failed. No data received.';
			
			return $this->doReturn($result);
		}
		
		if ($response->code != 200)
		{
			if ($this->try_count == 1 && !empty($this->remote_host_secured))
			{
				// Try again using secured host
				$this->remote_host = $this->remote_host_secured;
				
				return $this->_validate($query);
			}
			
			$result['msg'] = 'Error:' . $response->code . '. Verification failed. Invalid data received.';
			
			return $this->doReturn($result);
		}
		
		// Create a shortcut to response body
		$return = $response->body;
		
		if (empty($return))
		{
			$result['msg'] = 'Could not fetch data from remote server.';
		}
		else
		{
			if (is_string($return) && strpos($return, '.well-known') && strpos($return, 'captcha'))
			{
				// Server running behind a proxy
				// Verification failed but we will allow it to save the DLK
				$result['error'] = false;
				
				$return = array();
				$date   = JFactory::getDate();
				
				$return['valid']        = 1;
				$return['access']       = 0;
				$return['message']      = $errorMessage ? $errorMessage : 'Your server is running behind a proxy and it can not download live updates.';
				$return['dlk']          = $query['dlk'];
				$return['data']         = base64_encode($query['host'] . '|*|' . $query['dlk'] . '|*|' . $query['uname'] . '|*|' . intval($return['access']). '|*|' . $date->toSql());
				$return['last_checked'] = JHtml::_('date', $date, 'F d, Y H:i:s');
			}
			elseif (is_string($return))
			{
				$return = @json_decode($return, true);
				
				if (!is_array($return) || empty($return))
				{
					$result['msg'] = 'Verification failed. Invalid data received.';
				}
				else
				{
					$host   = isset($return['host']) ? $return['host'] : '';
					$_pkey  = isset($return['_pkey']) ? $return['_pkey'] : '';
					
					if ($host != $query['host'])
					{
						$result['msg'] = 'Host name verification failed.';
					}
					elseif ($_pkey != $query['_pkey'])
					{
						$result['msg'] = 'Verification failed.';
					}
					elseif ((empty($return['valid']) || empty($return['access'])) && !empty($return['message']))
					{
						$result['msg'] = $return['message'];
					}
					else
					{
						// Verification success
						$result['error'] = false;
					}
					
					$date = JFactory::getDate();
					
					$return['dlk']  = $query['dlk'];
					$return['data'] = base64_encode($host . '|*|' . $query['dlk'] . '|*|' . $query['uname'] . '|*|' . intval($return['access']). '|*|' . $date->toSql());
					$return['last_checked'] = JHtml::_('date', $date, 'F d, Y H:i:s');
				}
			}
			else
			{
				$result['msg'] = 'Verification failed. Invalid data format.';
			}
		}
		
		if (!$result['error'] && !empty($return['dlk']))
		{
			if (!$this->save($return['data']))
			{
				$result['error'] = true;
				$result['msg'] = $this->getError();
			}
			
			if (!$result['error'])
			{
				if (!$this->refreshUpdates($return['dlk']))
				{
					$result['error'] = true;
					$result['msg'] = $this->getError();
				}
			}
		}
		
		$result['return'] = $return;
		
		return $this->doReturn($result);
	}
	
	protected function save($data)
	{
		if (!is_object($this->framework) || empty($this->framework->name))
		{
			$this->setError('Framework plugin information not found.');
			return false;
		}
		
		$extension_id = (int) $this->package->extension_id;
		
		if (!$extension_id)
		{
			$this->setError('Package extension id not found.');
			return false;
		}
		
		$key    = 'pack_id_' . $extension_id;
		$params = ($this->params instanceof JRegistry) ? $this->params : new JRegistry;
		$dlks   = (array) $params->get('dlks', array());
		
		$temp = new JRegistry;
		$temp->loadArray($dlks);
		$temp->set($key, $data);
		
		$params->set('dlks', $temp->toArray());
		
		// Now save the new params in the framework
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		
		$query->update($db->quoteName('#__extensions'))
		      ->set($db->quoteName('params') . ' = ' . $db->quote($params->toString()))
		      ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
		      ->where($db->quoteName('element') . ' = ' . $db->quote($this->framework->name))
		      ->where($db->quoteName('folder') . ' = ' . $db->quote('system'));
		
		try
		{
			$db->setQuery($query);
			$db->execute();
		}
		catch(Exception $e)
		{
			$this->setError($e->getMessage());
			return false;
		}
		
		$this->cleanCache('_system');
		$this->cleanCache('com_plugins');
		
		return true;
	}
	
	protected function cleanCache($group, $client_id = 0)
	{
		$conf = JFactory::getConfig();

		$options = array(
			'defaultgroup' => $group,
			'cachebase'    => ($client_id) ? JPATH_ADMINISTRATOR . '/cache' : $conf->get('cache_path', JPATH_SITE . '/cache')
		);
		
		try
		{
			$cache = JCache::getInstance('callback', $options);
			$cache->clean();
		}
		catch (\JCacheException $exception)
		{
			$options['result'] = false;
		}

		// Trigger the onContentCleanCache event.
		if (version_compare(JVERSION, '4.0.0', 'ge'))
		{
			JFactory::getApplication()->triggerEvent('onExtensionCleanCache', $options);
		}
		else
		{
			JEventDispatcher::getInstance()->trigger('onExtensionCleanCache', $options);
		}
	}

	protected static function getPackages()
	{
		if (self::$packages === null)
		{
			self::$packages = array();
			$framework      = JPluginHelper::getPlugin('system', 'vpframework');
			$params         = new JRegistry;
			$db             = JFactory::getDbo();
			$query          = $db->getQuery(true);
			
			if (!empty($framework->params))
			{
				if (is_string($framework->params))
				{
					$params->loadString($framework->params);
				}
				elseif (is_array($framework->params))
				{
					$params->loadArray($framework->params);
				}
				elseif (is_object($framework->params))
				{
					$params = $framework->params;
				}
			}
			
			$dlks = (array) $params->get('dlks', array());
			
			foreach ($dlks as $key => $value)
			{
				$key = (string) $key;
				
				if (strpos($key, 'pack_id_') === false || strlen($key) < 9)
				{
					continue;
				}
				
				$pack_id = str_replace('pack_id_', '', $key);
				
				if (!is_numeric($pack_id))
				{
					continue;
				}
				
				$query->clear()
				      ->select($db->quoteName('element'))
				      ->from($db->quoteName('#__extensions'))
				      ->where($db->quoteName('extension_id') . ' = ' . (int) $pack_id);
				$db->setQuery($query);
				
				try
				{
					$package_name = $db->loadResult();
				}
				catch (Exception $e) 
				{
					// Some error occured
					continue;
				}
				
				if (empty($package_name))
				{
					continue;
				}
				
				$data = self::decodeData($value);
				
				if (empty($data['dlk']))
				{
					continue;
				}
				
				self::$packages[$package_name] = array('extension_id' => $pack_id, 'dlk' => $data['dlk']);
			}
		}
		
		return self::$packages;
	}
	
	
	public static function decodeData($string, $renewHost = false)
	{
		$data = array('host' => '', 'dlk' => '', 'uname' => '', 'access' => 0, 'last_checked' => '');

		if (empty($string) || !is_string($string))
		{
			return $data;
		}
		
		$string = @base64_decode($string);
		
		if (empty($string) || !is_string($string))
		{
			return $data;
		}
		
		$parts  = !empty($string) && strpos($string, '|*|') !== false ? explode('|*|', $string) : array();
		$newHost = self::getHost();
		
		if ($renewHost)
		{
			$data['host']         = $newHost;
			$data['dlk']          = isset($parts[1]) ? self::cleanDlk($parts[1]) : '';
			$data['uname']        = isset($parts[2]) ? $parts[2] : '';
			$data['access']       = isset($parts[3]) ? intval($parts[3]) : 0;
			$data['last_checked'] = isset($parts[4]) ? $parts[4] : null;
		}
		else
		{
			$host = '';
			
			if (!empty($parts[0]))
			{
				$host = base64_decode($parts[0]);
				
				if ($host === false)
				{
					$host = $parts[0];
				}
			}
			
			$prefix = 'www.';
			
			if (!empty($host) && substr($host, 0, strlen($prefix)) == $prefix)
			{
				$host = substr($host, strlen($prefix));
			}

			if (!empty($newHost) && substr($newHost, 0, strlen($prefix)) == $prefix)
			{
				$newHost = substr($newHost, strlen($prefix));
			}

			if (!empty($newHost) && $host == $newHost)
			{
				$data['host']         = $newHost;
				$data['dlk']          = isset($parts[1]) ? self::cleanDlk($parts[1]) : '';
				$data['uname']        = isset($parts[2]) ? $parts[2] : '';
				$data['access']       = isset($parts[3]) ? intval($parts[3]) : 0;
				$data['last_checked'] = isset($parts[4]) ? $parts[4] : null;
			}
		}
		
		return $data;
	}
	
	protected static function getHost($host = null)
	{
		$host = $host ? $host : JUri::root();
		
		if (empty($host))
		{
			return '';
		}
		
		$parts = parse_url($host);
		
		$result = '';
		
		if (!empty($parts['host']))
		{
			$result .= $parts['host'];
		}
		
		if (!empty($parts['path']))
		{
			$result .= $parts['path'];
		}
		
		return $result;
	}
	
	protected function refreshUpdates($dlk, $clear = false)
	{
		$extension_id = $this->input->getInt('extension_id', 0);

		if (!$extension_id || (empty($dlk) && !$clear))
		{
			return false;
		}
		
		if ((!$dlk = self::cleanDlk($dlk)) && !$clear)
		{
			return false;
		}
		
		$db    = JFactory::getDbo();
		$query = $db->getQuery(true);
		
		if ($clear)
		{
			$extra_query = '';
		}
		else
		{
			$extra_query = 'dlid=' . $dlk;
		}
		
		// Get the update sites for current extension
		$query->select('update_site_id')
		      ->from('#__update_sites_extensions')
		      ->where('extension_id = ' . (int) $extension_id);
		
		try
		{
			$db->setQuery($query);
			$updateSiteIDs = $db->loadColumn();
		}
		catch (Exception $e)
		{
			$this->setError($e->getMessage());
			return false;
		}
		
		if (count($updateSiteIDs))
		{
			foreach ($updateSiteIDs as $id)
			{
				$query->clear()
				      ->select('*')
				      ->from('#__update_sites')
				      ->where('update_site_id = ' . (int) $id);

				try
				{
					$db->setQuery($query);
					$updateSite = $db->loadObject();
				}
				catch (Exception $e)
				{
					$this->setError($e->getMessage());
					return false;
				}
				
				if (!is_object($updateSite))
				{
					continue;
				}

				// Do we have the extra_query property (J 3.2+) and does it match?
				if (property_exists($updateSite, 'extra_query'))
				{
					if ($updateSite->extra_query == $extra_query)
					{
						continue;
					}
				}
				else
				{
					// Joomla! 3.1 or earlier. Updates may or may not work.
					continue;
				}
				
				$updateSite->update_site_id = $id;
				$updateSite->extra_query    = $extra_query;
				
				try
				{
					$db->updateObject('#__update_sites', $updateSite, 'update_site_id', true);
				}
				catch (Exception $e)
				{
					$this->setError($e->getMessage());
					return false;
				}
			}
		}
		
		return true;
	}
	
	protected function loadInfo()
	{
		$app = JFactory::getApplication();
		
		if ($app->isClient('administrator'))
		{
			$adminHelper = VPFrameworkAdmin::getInstance();
			$template    = $adminHelper->getTemplate();
		}
		else
		{
			$helper   = plgSystemVPFrameworkHelper::getInstance();
			$template = $helper->getTemplate();
		}

		$this->template = $template;

		if (!empty($template->template))
		{
			$package_element = strtolower('pkg_' . $template->template);
			
			$db    = JFactory::getDbo();
			$query = $db->getQuery(true);
			
			$query->select('*')
			      ->from($db->quoteName('#__extensions'))
			      ->where($db->quoteName('type') . ' = ' . $db->quote('package'))
			      ->where($db->quoteName('element') . ' = ' . $db->quote($package_element))
			      ->where($db->quoteName('enabled') . ' = ' . $db->quote('1'));
			
			$db->setQuery($query);
			$package = $db->loadObject();
			
			if (!empty($package) && is_object($package))
			{
				if (!empty($package->manifest_cache) && is_string($package->manifest_cache))
				{
					$temp = new JRegistry;
					$temp->loadString($package->manifest_cache);
					$package->manifest_cache = $temp;
					
					if ($version = $package->manifest_cache->get('version', 0))
					{
						$package->version = $version;
						$this->package = $package;
					}
				}
				
				if (!empty($package->params) && is_string($package->params))
				{
					$temp = new JRegistry;
					$temp->loadString($package->params);
					$package->params = $temp;
					$pid = (int) $package->params->get('pid', 0);
					
					if ($pid > 0)
					{
						$this->pid = $pid;
					}
					else
					{
						$this->package = null;
					}
				}
			}
		}
	}
	
	/**
	* Method to return JSON object values with proper header
	* 
	* @param arry $message Array to be return as JSON object
	* 
	* @return void
	*/
	protected function doReturn($output) 
	{
		$app     = JFactory::getApplication();
		$obLevel = ob_get_level();
		
		if ($obLevel)
		{
			while ($obLevel > 0)
			{
				@ob_end_clean();
				$obLevel --;
			}
		}
		else
		{
			@ob_clean();
		}
		
		header('Content-type: application/text');
		header('Content-type: application/json');
		header('Cache-Control: public,max-age=1,must-revalidate');
		header('Expires: ' . gmdate('D, d M Y H:i:s', ($_SERVER['REQUEST_TIME'] + 1)) . ' GMT');
		header('Last-modified: ' . gmdate('D, d M Y H:i:s', $_SERVER['REQUEST_TIME']) . ' GMT');
		
		if (function_exists('header_remove')) 
		{
			@header_remove('Pragma');
		}
		
		$this->time_end = microtime(true);
		$execution_time = ($this->time_end - $this->time_start);
		
		if ($execution_time < 1)
		{
			$execution_time = number_format(($execution_time * 1000), 2, '.', ',') . ' ms';
		}
		else
		{
			$execution_time = number_format($execution_time, 6, '.', ',') . ' s';
		}
		
		$output = (array) $output;
		$output['execution_time'] = $execution_time;
		
		echo json_encode($output);
		
		flush();
		$app->close();
	}
	
	protected static function cleanDlk($dlk)
	{
		if (empty($dlk))
		{
			return '';
		}
		
		$dlk = trim($dlk);

		if (empty($dlk))
		{
			return '';
		}
		
		// Is the Download Key too short?
		if (strlen($dlk) < 32)
		{
			return '';
		}
		
		if (strlen($dlk) > 32)
		{
			$dlk = substr($dlk, 0, 32);
		}
		
		$dlk = preg_replace("/[^a-zA-Z0-9]+/", "", $dlk);
		
		return $dlk;
	}
}