<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

class VPFrameworkOptimizerBuilder
{
	protected $params;
	protected $parser;
	protected $css_parser;
	protected $http;
	protected $cache;
	protected $hash;
	
	protected static $tab;
	protected static $lineEnd;
	protected static $loadFunctionAdded = false;
	
	public function __construct($params, $parser, $css_parser, $cache, $hash)
	{
		$this->params     = $params;
		$this->parser     = $parser;
		$this->css_parser = $css_parser;
		$this->cache      = $cache;
		$this->hash       = $hash;
		$this->http       = VPFrameworkHttp::getInstance();
		
		$doc              = JFactory::getDocument();
		self::$tab        = $doc->_getTab();
		self::$lineEnd    = $doc->_getLineEnd();
	}
	
	public function getContents($type, $links)
	{
		$hasAsync = false;
		$contents = '';
		$count    = count($links);
		
		foreach ($links as $attribs)
		{
			if ((!empty($attribs['exclude']) && !empty($attribs['element']) && $count == 1))
			{
				// Return the original element in case of exclusion
				return array('element' => $attribs['element']);
			}
			
			$url     = isset($attribs['url']) ? $attribs['url'] : null;
			$content = isset($attribs['content']) ? $attribs['content'] : null;
			
			if ($type == 'js' && !empty($url) && (!empty($attribs['async']) || !empty($attribs['defer'])) && !$this->noAsync($url))
			{
				$loadAsync = 'VPLoadScript("' . $url . '", function(){/**Load Async**/});';
				$contents .= $this->addTryCatch($url, $loadAsync);
				
				$hasAsync = true;
			}
			elseif (!empty($url) && empty($content))
			{
				$callback  = array($this, 'getContent');
				$arguments = array($type, $attribs, true);
				$cache_id  = md5($this->hash . $url . (!empty($attribs['hash']) ? serialize($attribs['hash']) : ''));
				
				$contents .= $this->cache->get($callback, $arguments, $cache_id);
			}
			elseif (!empty($content))
			{
				$contents .= $this->getContent($type, $attribs, false);
			}
		}
		
		if ($type == 'js' && $hasAsync && !self::$loadFunctionAdded)
		{
			$contents = $this->getLoadScript() . $contents;
			self::$loadFunctionAdded = true;
		}

		return $contents;
	}
	
	public function getContent($type, $context, $minify = false)
	{
		$url     = isset($context['url']) ? $context['url'] : null;
		$content = isset($context['content']) ? $context['content'] : '';
		
		if (!empty($url) && empty($content))
		{
			$path    = VPFrameworkUrl::getFilePath($url);
			$content = $this->http->getFileContents($path);
		}
		
		if (!$url)
		{
			$url = 'Inline';
		}
		
		if ($type == 'js' || $type == 'js_inline')
		{
			$content = $this->addTryCatch($url, $content);
		}
		
		if ($type == 'css' || $type == 'css_inline')
		{
			$importContent = preg_replace('#@import\s(?:url\()?[\'"]([^\'"]+)[\'"](?:\))?#', '@import url($1)', $content);
			
			if (is_null($importContent))
			{
				//TODO: Log find "@imports" in the CSS file failed
			}
			else
			{
				$content = $importContent;
			}
			
			unset($importContent);
			
			$content = $this->css_parser->addRightBrace($content);
			
			$content = $this->css_parser->fixUrl($content, $context);
			
			$content = $this->replaceImports($content, $context);
			
			$media   = !empty($context['media']) ? $context['media'] : '';
			$content = $this->css_parser->manageMediaQueries($content, $media);
		}
		
		if ($minify)
		{
			$content = self::minifyContent($type, $content, $url);
			$content = $this->prepareComments($content);
		}
		
		return $content;
	}
	
	protected function addTryCatch($url, $content)
	{
		//$prefix  = '|"VP_COMMENT_START URL [' . $url . '] not found VP_COMMENT_END"|;';
		//$content = $prefix . self::$lineEnd . $content;
		
		if ($this->params->get('optimizer_add_try_catch_js', 1))
		{
			$content  = 'try {' . $content . '} catch (e) {';
			$content .= 'console.error(\'Error in file: ';
			$content .= $url;
			$content .= '; Error:\' + e.message);' . '};' . self::$lineEnd;
		}
		
		return $content;
	}
	
	protected function prepareComments($content)
	{
		$old = array(
			'|"VP_COMMENT_START',
			'VP_COMMENT_END"|',
			'|"VP_COMMENT_IMPORT_START',
			'|"VP_LINE_END"|'
		);
		
		$new = array(
			self::$lineEnd . '/***! ',
			' !***/' . self::$lineEnd . self::$lineEnd,
			self::$lineEnd . self::$lineEnd . '/***! @import url',
			self::$lineEnd
		);
		
		return str_replace($old, $new, $content);
	}
	
	
	public static function minifyContent($type, $content, $identifier)
	{
		static $params = null;
		
		$type = strtolower($type);
		
		if ($params === null)
		{
			$params = plgSystemVPFrameworkHelper::getTemplate()->params;
		}
		
		if ($params->get('optimizer_minify_' . $type, 1) && $content)
		{
			if (!class_exists('VPFrameworkCompressor'))
			{
				require VPF_PLUGINPATH . '/helpers/compressors/base.php';
			}
			
			$debug = false;
			$timelimit = 0;
			$debug_regex_name = null;
			
			$css_compressor  = VPFrameworkCompressor::getInstance('css', $debug, $timelimit, $debug_regex_name);
			$js_compressor   = VPFrameworkCompressor::getInstance('js', $debug, $timelimit, $debug_regex_name);
			$html_compressor = VPFrameworkCompressor::getInstance('html', $debug, $timelimit, $debug_regex_name);
			
			if ($type == 'html')
			{
				// Set compression strength
				$html_compressor->setStrength($params->get('optimizer_compress_html_strength', 4));
				
				$minifiedContent = $html_compressor->proccess($content, $js_compressor, $css_compressor);
			}
			elseif ($type == 'css')
			{
				$minifiedContent = $css_compressor->proccess($content);
			}
			elseif ($type == 'js')
			{
				$minifiedContent = $js_compressor->proccess($content);
			}
			
			if (empty($minifiedContent))
			{
				//TODO: Log error with identifier
				
			}
			else
			{
				$content = $minifiedContent;
			}
		}

		return $content;
	}
	
	protected function getLoadScript()
	{
		$js = '
		function VPLoadScript(url, callback) {
			var script = document.createElement("script");
			
			script.type = "text/javascript";
			script.async = true;

			if (script.readyState){  //IE
				script.onreadystatechange = function() {
					if (script.readyState == "loaded" || script.readyState == "complete") {
						script.onreadystatechange = null;
						callback();
					}
				};
			} else {  //Others
				script.onload = function() {
					callback();
				};
			}

			script.src = url;
			document.getElementsByTagName("head")[0].appendChild(script);
		};';

		return $this->minifyContent('js', $js, 'loadScript') . self::$lineEnd;
	}
	
	protected function replaceImports($content)
	{
		if ($this->params->get('optimizer_replace_imports', 1))
		{
			$url_regex = $this->css_parser->url;
			$regex     = "#(?>@?[^@'\"/]*+(?:{$url_regex}|/|\()?)*?\K(?:@import\s*+(?:url\()?['\"]?([^\)'\"]+)['\"]?(?:\))?\s*+([^;]*);|\K$)#";

			$replacedContents = preg_replace_callback($regex, array(__CLASS__, 'getImportFileContents'), $content);

			if (is_null($replacedContents))
			{
				//TODO:: Log failed to get the contents of the file that was imported into the document by the "@import" rule'
			}
			else
			{
				$content = $replacedContents;
			}
		}

		return $content;
	}
	
	protected function getImportFileContents($matches)
	{
		if (empty($matches[1]) || preg_match('#^(?>\(|/\*)#', $matches[0]) ||
		    !$this->http->hasAdapter() || (VPFrameworkUrl::isSSL($matches[1]) && !extension_loaded('openssl')) ||
		    (!VPFrameworkUrl::isHttpScheme($matches[1])))
		{
			return $matches[0];
		}

		if ($this->params->get('optimizer_skip_googlefont', 1) && strpos($matches[1], 'fonts.googleapis.com') !== false)
		{
			return $matches[0];
		}

		if ($this->parser->isDuplicated($matches[1]))
		{
			return '';
		}
		
		$asset = array();
		
		$asset['url'] = $matches[1];
		$asset['media'] = $matches[2];
		
		$links = array($asset);

		$contents = $this->getContents('css', $links);


		if ($contents === false)
		{
			return $matches[0];
		}

		return $contents;
	}
	
	protected function noAsync($url)
	{
		if (strpos($url, 'com_virtuemart') !== false || strpos($url, 'cvfind.js') !== false || strpos($url, 'vmprices.js') !== false || strpos($url, 'vmsite.js') !== false)
		{
			return true;
		}
		
		return false;
	}
}