<?php
/**
 *---------------------------------------------------------------------------------------
 * @package       VirtuePlanet Framework for Joomla!
 *---------------------------------------------------------------------------------------
 * @copyright     Copyright (C) 2012-2024 VirtuePlanet Services LLP. All rights reserved.
 * @license       GNU General Public License version 2 or later; see LICENSE.txt
 * @authors       Abhishek Das
 * @email         info@virtueplanet.com
 * @link          https://www.virtueplanet.com
 *---------------------------------------------------------------------------------------
 */
defined('_JEXEC') or die;

class VPFrameworkOptimizerCssparser
{
	protected $double_quote  = '"(?>(?:\\\\.)?[^\\\\"]*+)+?(?:"|(?=$))';
	protected $single_quote  = "'(?>(?:\\\\.)?[^\\\\']*+)+?(?:'|(?=$))";
	protected $comment_block = '/\*(?>[^/\*]++|//|\*(?!/)|(?<!\*)/)*+\*/';
	protected $comment_line  = '//[^\r\n]*+';
	protected $uri           = '(?<=url)\([^)]*+\)';
	
	public $tab;
	public $lineEnd;
	public $params;
	public $backEnd;
	public $escape;
	public $url;

	public function __construct($params = null, $backEnd = false)
	{
		if ($params && is_object($params))
		{
			$this->params = $params;
		}
		else
		{
			$this->params = plgSystemVPFrameworkHelper::getTemplate()->params;
		}
		
		$doc = JFactory::getDocument();
		
		$this->tab     = $doc->_getTab();
		$this->lineEnd = $doc->_getLineEnd();
		$this->backEnd = $backEnd;
		
		$escape = $this->double_quote . '|' . $this->single_quote . '|' . $this->comment_block . '|' . $this->comment_line;
		
		$this->escape = "(?<!\\\\)(?:$escape)|[\'\"/]";
		$this->url    = '(?<!\\\\)(?:' . $this->uri . '|' . $escape . ')|[\'\"/(]';
	}

	public function manageMediaQueries($content, $elementMedia = '')
	{
		if ($this->backEnd)
		{
			return $content;
		}

		if (!empty($elementMedia))
		{
			$that  = $this;
			$regex = "#(?>@?[^@'\"/(]*+(?:{$this->url})?)*?\K(?:@media ([^{]*+)|\K$)#i";
			
			$content = preg_replace_callback($regex, function($matches) use ($elementMedia, $that) {
				return $that->mergeMediaQueries($matches, $elementMedia);
			}, $content);
			
			$nested_rule_regex = '@[^{};]++({(?>[^{}]++|(?1))*+})';
			
			$regex = "#(?>(?:\|\"[^|]++(?<=\")\||$nested_rule_regex)\s*+)*\K" .
			         "(?>(?:$this->url|/|\(|@(?![^{};]++(?1)))?(?:[^|@'\"/(]*+|$))*+#i";
			
			$content = preg_replace($regex, '@media ' . $elementMedia . ' {' . $this->lineEnd . '$0' . $this->lineEnd . '}', trim($content));
			$content = preg_replace("#(?>@?[^@'\"/(]*+(?:{$this->url})?)*?\K(?:@media[^{]*+{((?>\s*+|$this->escape)++)}|$)#i", '$1', $content);
		}

		return $content;
	}


	public function mergeMediaQueries($matches, $elementMedia)
	{
		if (empty($matches[1]) || preg_match('#^(?>\(|/(?>/|\*))#', $matches[0]))
		{
			return $matches[0];
		}

		return '@media ' . $this->_mergeMediaQueries($elementMedia, trim($matches[1]));
	}

	protected function _mergeMediaQueries($parentMediaQueries, $childMediaQueries)
	{
		$parentMediaQueries = preg_split('#\s++or\s++|,#i', $parentMediaQueries);
		$childMediaQueries  = preg_split('#\s++or\s++|,#i', $childMediaQueries);

		$mediaQueries = array();

		foreach ($parentMediaQueries as $parentQuery)
		{
			$parentQuery = $this->parseMediaQuery(trim($parentQuery));

			foreach ($childMediaQueries as $childQuery)
			{
				$mediaQuery = '';
				$childQuery = $this->parseMediaQuery(trim($childQuery));

				if ($parentQuery['keyword'] == 'only' || $childQuery['keyword'] == 'only')
				{
					$mediaQuery .= 'only ';
				}

				if ($parentQuery['keyword'] == 'not' && $childQuery['keyword'] == '')
				{
					if ($parentQuery['media_type'] == 'all')
					{
						$mediaQuery .= '(not ' . $parentQuery['media_type'] . ')';
					}
					elseif ($parentQuery['media_type'] == $childQuery['media_type'])
					{
						$mediaQuery .= '(not ' . $parentQuery['media_type'] . ') and ' . $childQuery['media_type'];
					}
					else
					{
						$mediaQuery .= $childQuery['media_type'];
					}
				}
				elseif ($parentQuery['keyword'] == '' && $childQuery['keyword'] == 'not')
				{
					if ($childQuery['media_type'] == 'all')
					{
						$mediaQuery .= '(not ' . $childQuery['media_type'] . ')';
					}
					elseif ($parentQuery['media_type'] == $childQuery['media_type'])
					{
						$mediaQuery .= $parentQuery['media_type'] . ' and (not ' . $childQuery['media_type'] . ')';
					}
					else
					{
						$mediaQuery .= $childQuery['media_type'];
					}
				}
				elseif ($parentQuery['keyword'] == 'not' && $childQuery['keyword'] == 'not')
				{
					$mediaQuery .= 'not ' . $childQuery['keyword'];
				}
				else
				{
					if ($parentQuery['media_type'] == $childQuery['media_type'] || $parentQuery['media_type'] == 'all')
					{
						$mediaQuery .= $childQuery['media_type'];
					}
					elseif ($childQuery['media_type'] == 'all')
					{
						$mediaQuery .= $parentQuery['media_type'];
					}
					else
					{
						$mediaQuery .= $parentQuery['media_type'] . ' and ' . $childQuery['media_type'];
					}
				}

				if (isset($parentQuery['expression']))
				{
					$mediaQuery .= ' and ' . $parentQuery['expression'];
				}

				if (isset($childQuery['expression']))
				{
					$mediaQuery .= ' and ' . $childQuery['expression'];
				}

				$mediaQueries[] = $mediaQuery;
			}
		}

		return implode(', ', $mediaQueries);
	}

	protected function parseMediaQuery($mediaQuery)
	{
		$parts = array();

		$mediaQuery = preg_replace(array('#\(\s++#', '#\s++\)#'), array('(', ')'), $mediaQuery);
		
		preg_match('#(?:\(?(not|only)\)?)?\s*+(?:\(?(all|aural|braille|handheld|print|projection|screen|tty|tv|embossed)\)?)?(?:\s++and\s++)?(.++)?#si', $mediaQuery, $matches);

		$parts['keyword'] = isset($matches[1]) ? strtolower($matches[1]) : '';

		if (isset($matches[2]) && $matches[2] != '')
		{
			$parts['media_type'] = strtolower($matches[2]);
		}
		else
		{
			$parts['media_type'] = 'all';
		}

		if (isset($matches[3]) && $matches[3] != '')
		{
			$parts['expression'] = $matches[3];
		}

		return $parts;
	}

	public function removeAtRules($content, $atRulesRegex, $sUrl = array('url' => 'CSS'))
	{
		if (preg_match_all($atRulesRegex, $content, $matches) === false)
		{
			return $content;
		}

		$match = array_filter($matches[0]);

		if (!empty($match))
		{
			$match = array_unique($match);
			$atRules = implode($this->lineEnd, $match);
			$contentReplaced = str_replace($match, '', $content);
			$content = $atRules . $this->lineEnd . $this->lineEnd . $contentReplaced;
		}

		return $content;
	}

	public function fixUrl($content, $attribs)
	{
		$that = $this;
		$regex = "#(?>[(]?[^('/\"]*+(?:{$this->escape}|/)?)*?(?:(?<=url)\(\s*+\K['\"]?((?<!['\"])[^\s)]*+|(?<!')[^\"]*+|[^']*+)['\"]?|\K$)#i";

		$fixedContent = preg_replace_callback($regex, function($matches) use ($attribs, $that) {
			return $that->_fixUrl($matches, $attribs);
		}, $content);

		if (is_null($fixedContent))
		{
			throw new Exception('The plugin failed to correct the url of the background images');
		}

		$content = $fixedContent;

		return $content;
	}

	public function _fixUrl($matches, $attribs)
	{
		if (empty($matches[1]) || preg_match('#^(?:\(|/\*)#', $matches[0]))
		{
			return $matches[0];
		}

		$fileUrl    = $matches[1];
		$cssFileUrl = empty($attribs['url']) ? '' : $attribs['url'];

		if (VPFrameworkUrl::isHttpScheme($fileUrl))
		{
			if ((VPFrameworkUrl::isInternal($cssFileUrl) || $cssFileUrl == '') && VPFrameworkUrl::isInternal($fileUrl))
			{
				$fileUrl = VPFrameworkUrl::toRootRelative($fileUrl, $cssFileUrl);
				$fileUri = clone VPFrameworkUrl::getInstance($fileUrl);

				$staticFiles = $this->params->get('optimizer_staticfiles', array('css', 'js', 'jpe?g', 'gif', 'png', 'ico', 'bmp', 'pdf', 'tiff?', 'docx?'));
				
				if ($key = array_search('css', $staticFiles))
				{
					unset($staticFiles[$key]);
				}
				
				if ($key = array_search('js', $staticFiles))
				{
					unset($staticFiles[$key]);
				}
				
				$staticFiles = implode('|', $staticFiles);

				$fontFiles = self::fontFiles();
				$fontFiles = implode('|', $fontFiles);

				if (preg_match('#\.(?>' . $staticFiles . ')#', $fileUri->getPath()))
				{
					$fileUrl = VPFrameworkOptimizer::setCdn($fileUri->toString(array('path')), $this->params);
				}
				elseif (VPFrameworkOptimizer::cdnEnabled($this->params) && preg_match('#\.(?>' . $fontFiles . ')#', $fileUri->getPath()))
				{
					if ($this->params->get('optimizer_cdn_include_fonts', 0))
					{
						$fileUrl = VPFrameworkOptimizer::setCdn($fileUri->toString(array('path')), $this->params);
					}
					else
					{
						$oUri = clone VPFrameworkUrl::getInstance();
						$fileUrl = '//' . $oUri->toString(array('host', 'port')) . $fileUri->toString(array('path'));
					}
				}
			}
			else
			{
				if (!VPFrameworkUrl::isAbsolute($fileUrl))
				{
					$fileUrl = VPFrameworkUrl::toAbsolute($fileUrl, $cssFileUrl);
				}
			}
		}

		$fileUrl = preg_match('#(?<!\\\\)[\s\'"(),]#', $fileUrl) ? '"' . $fileUrl . '"' : $fileUrl;

		return $fileUrl;
	}

	/**
	* Sorts @import and @charset as according to w3C <http://www.w3.org/TR/CSS2/cascade.html> Section 6.3
	*
	* @param  string $css  Combined css
	* 
	* @return string CSS with @import and @charset properly sorted
	*/
	public function sortImports($css)
	{
		$regex   = "#(?>@?[^@('\"/]*+(?:{$this->url}|/|\()?)*?\K(?:@media\s([^{]++)({(?>[^{}]++|(?2))*+})|\K$)#i";
		$imports = preg_replace_callback($regex, array(__CLASS__, '_sortImports'), $css);

		if (is_null($imports))
		{
			return $css;
		}

		$css = $imports;

		$css = preg_replace('#@charset[^;}]++;?#i', '', $css);
		$css = $this->removeAtRules($css, '#(?>[/@]?[^/@]*+(?:/\*(?>\*?[^\*]*+)*?\*/)?)*?\K(?:@import[^;}]++;?|\K$)#i');

		return $css;
	}

	protected function _sortImports($matches)
	{
		if (empty($matches[1]) || preg_match('#^(?>\(|/(?>/|\*))#', $matches[0]))
		{
			return $matches[0];
		}

		$media = $matches[1];

		$imports = preg_replace_callback('#(@import\surl\([^)]++\))([^;}]*+);?#', function($iMatches) use ($media) {
			if (!empty($iMatches[2]))
			{
				return $iMatches[1] . ' ' . $this->combineMediaQueries($media, $iMatches[2]) . ';';
			}
			else
			{
				return $iMatches[1] . ' ' . $media . ';';
			}
		}, $matches[2]);

		return str_replace($matches[2], $imports, $matches[0]);
	}

	public function addRightBrace($css)
	{
		$result = '';

		preg_replace_callback("#(?>[^{}'\"/(]*+(?:{$this->url})?)+?(?:(?<b>{(?>[^{}]++|(?&b))*+})|\||)#", function($matches) use (&$result) {
			$result .= $matches[0];
			return;
		}, rtrim($css) . '}}');

		return $result;
	}


	public static function cssRulesRegex()
	{
		$comments = $this->comment_block . '|' . $this->comment_line;
		$regex    = "(?:\s*+(?>$comments)\s*+)*+\K" .
		            "((?>[^{}@/]*+(?:/(?![*/]))?)*?)(?>{[^{}]*+}|(@[^{};]*+)(?>({((?>[^{}]++|(?3))*+)})|;?)|$)";
		
		return $regex;
	}

	public static function staticFiles()
	{
		$extensions = array(
			'css', 'js',
			'jpe?g', 'gif',
			'ico', 'png',
			'bmp', 'pdf',
			'pls', 'tif',
			'mid', 'doc',
		);

		return $extensions;
	}

	public static function fontFiles()
	{
		$extensions = array(
			'woff', 'ttf',
			'otf', 'svg',
			'eot'
		);

		return $extensions;
	}
}
